<?php

namespace App\Repositories\Payment\PaymentGateways;

use App\Models\ChargeWallet;
use App\Repositories\Order\OrderRepository;
use App\Repositories\Setting\SettingRepository;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class AsanPardakhtGateway implements PaymentGatewayInterface
{
    protected string $merchantId;
    protected string $username;
    protected string $password;
    protected string $baseUrl;
    protected string $callbackUrl;

    public function __construct()
    {
        $this->merchantId = env('ASANPARDAKHT_MERCHANT_ID');
        $this->username = env('ASANPARDAKHT_USERNAME');
        $this->password = env('ASANPARDAKHT_PASSWORD');
        $this->baseUrl = env('ASANPARDAKHT_BASE_URL', 'https://ipg.asanpardakht.ir/v1');
        $this->callbackUrl = env('ASANPARDAKHT_CALLBACK_URL', route('payConfirm', ['gateway_id' => 7]));
    }

    /**
     * ایجاد درخواست پرداخت
     *
     * @param string $paymentCode
     * @param int $amount
     * @param string|null $callbackRoute
     * @param int|null $orderId
     * @return array
     */
    public function setPay($paymentCode, $amount, $callbackRoute = null, $orderId = null): array
    {
        try {
            if ($callbackRoute == null) {
                $callbackRoute = $this->callbackUrl;
            }

            // دریافت تنظیمات از دیتابیس
            $merchantSetting = app(SettingRepository::class)->getSettingWithName('asanpardakht');
            if ($merchantSetting) {
                $this->merchantId = $merchantSetting['value'];
            }

            $payload = [
                'merchantConfigurationId' => $this->merchantId,
                'serviceTypeId' => 1, // نوع سرویس پرداخت
                'localInvoiceId' => $paymentCode,
                'amountInRials' => $amount,
                'localDate' => now()->format('Ymd'),
                'additionalData' => [
                    'mobileNumber' => '',
                    'emailAddress' => '',
                    'additionalData' => ''
                ],
                'callbackURL' => $callbackRoute,
                'paymentId' => $paymentCode,
                'additionalParameters' => []
            ];

            Log::info('AsanPardakht Pay Request', ['payload' => $payload]);

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])->post($this->baseUrl . '/Token', $payload);

            if ($response->successful()) {
                $result = $response->json();

                if (isset($result['status']) && $result['status'] == 0) {
                    return [
                        'status' => 200,
                        'message' => 'درخواست پرداخت با موفقیت ایجاد شد.',
                        'data' => [
                            'route' => $this->baseUrl . '/Payment/Index?token=' . $result['token'],
                            'codeGateway' => $result['token']
                        ]
                    ];
                } else {
                    Log::error('AsanPardakht Pay Error', ['response' => $result]);
                    return [
                        'status' => 500,
                        'message' => 'خطا در ایجاد درخواست پرداخت: ' . ($result['message'] ?? 'خطای نامشخص'),
                        'data' => null
                    ];
                }
            } else {
                Log::error('AsanPardakht HTTP Error', ['status' => $response->status(), 'body' => $response->body()]);
                return [
                    'status' => 500,
                    'message' => 'خطا در ارتباط با درگاه پرداخت آسان پرداخت',
                    'data' => null
                ];
            }

        } catch (\Exception $e) {
            Log::error('AsanPardakht Exception', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return [
                'status' => 500,
                'message' => 'خطا در پردازش درخواست پرداخت',
                'data' => null
            ];
        }
    }

    /**
     * تأیید پرداخت
     *
     * @param array $data
     * @return array
     */
    public function confirmPay($data): array
    {
        try {
            $token = $data['token'] ?? null;
            $resCode = $data['resCode'] ?? null;
            $saleOrderId = $data['saleOrderId'] ?? null;
            $refNum = $data['refNum'] ?? null;

            if (!$token) {
                return [
                    'status' => 400,
                    'message' => 'توکن پرداخت یافت نشد',
                    'data' => ['order' => null]
                ];
            }

            // پیدا کردن سفارش بر اساس توکن
            $order = app(OrderRepository::class)->getByCodeGateway($token);
            if (!$order) {
                return [
                    'status' => 404,
                    'message' => 'سفارش یافت نشد',
                    'data' => ['order' => null]
                ];
            }

            // بررسی وضعیت پرداخت
            if ($resCode != 0) {
                return [
                    'status' => 500,
                    'message' => 'پرداخت ناموفق بود. کد خطا: ' . $resCode,
                    'data' => ['order' => $order]
                ];
            }

            // تأیید پرداخت با درگاه
            $verifyResult = $this->verifyPayment($token, $saleOrderId, $refNum);

            if ($verifyResult['status'] == 200) {
                return [
                    'status' => 200,
                    'message' => 'پرداخت با موفقیت تأیید شد',
                    'data' => [
                        'order' => $order,
                        'refId' => $refNum
                    ]
                ];
            } else {
                return [
                    'status' => 500,
                    'message' => 'تأیید پرداخت ناموفق بود',
                    'data' => ['order' => $order]
                ];
            }

        } catch (\Exception $e) {
            Log::error('AsanPardakht Confirm Exception', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return [
                'status' => 500,
                'message' => 'خطا در تأیید پرداخت',
                'data' => ['order' => null]
            ];
        }
    }

    /**
     * تأیید پرداخت کیف پول
     *
     * @param array $data
     * @return array
     */
    public function confirmChargePay($data): array
    {
        try {
            $token = $data['token'] ?? null;
            $resCode = $data['resCode'] ?? null;
            $saleOrderId = $data['saleOrderId'] ?? null;
            $refNum = $data['refNum'] ?? null;

            if (!$token) {
                return [
                    'status' => 400,
                    'message' => 'توکن پرداخت یافت نشد',
                    'data' => ['order' => null]
                ];
            }

            // پیدا کردن تراکنش کیف پول
            $chargeWallet = ChargeWallet::where('payment_code_gateway', $token)->first();
            if (!$chargeWallet) {
                return [
                    'status' => 404,
                    'message' => 'تراکنش کیف پول یافت نشد',
                    'data' => ['order' => null]
                ];
            }

            // بررسی وضعیت پرداخت
            if ($resCode != 0) {
                return [
                    'status' => 500,
                    'message' => 'پرداخت کیف پول ناموفق بود. کد خطا: ' . $resCode,
                    'data' => ['order' => $chargeWallet]
                ];
            }

            // تأیید پرداخت با درگاه
            $verifyResult = $this->verifyPayment($token, $saleOrderId, $refNum);

            if ($verifyResult['status'] == 200) {
                return [
                    'status' => 200,
                    'message' => 'پرداخت کیف پول با موفقیت تأیید شد',
                    'data' => [
                        'order' => $chargeWallet,
                        'refId' => $refNum
                    ]
                ];
            } else {
                return [
                    'status' => 500,
                    'message' => 'تأیید پرداخت کیف پول ناموفق بود',
                    'data' => ['order' => $chargeWallet]
                ];
            }

        } catch (\Exception $e) {
            Log::error('AsanPardakht Confirm Charge Exception', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return [
                'status' => 500,
                'message' => 'خطا در تأیید پرداخت کیف پول',
                'data' => ['order' => null]
            ];
        }
    }

    /**
     * تأیید پرداخت با درگاه
     *
     * @param string $token
     * @param string $saleOrderId
     * @param string $refNum
     * @return array
     */
    protected function verifyPayment(string $token, string $saleOrderId, string $refNum): array
    {
        try {
            $payload = [
                'merchantConfigurationId' => $this->merchantId,
                'token' => $token,
                'saleOrderId' => $saleOrderId,
                'refNum' => $refNum
            ];

            Log::info('AsanPardakht Verify Request', ['payload' => $payload]);

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])->post($this->baseUrl . '/Verify', $payload);

            if ($response->successful()) {
                $result = $response->json();

                if (isset($result['status']) && $result['status'] == 0) {
                    return [
                        'status' => 200,
                        'message' => 'تأیید پرداخت موفق',
                        'data' => $result
                    ];
                } else {
                    Log::error('AsanPardakht Verify Error', ['response' => $result]);
                    return [
                        'status' => 500,
                        'message' => 'تأیید پرداخت ناموفق: ' . ($result['message'] ?? 'خطای نامشخص'),
                        'data' => $result
                    ];
                }
            } else {
                Log::error('AsanPardakht Verify HTTP Error', ['status' => $response->status(), 'body' => $response->body()]);
                return [
                    'status' => 500,
                    'message' => 'خطا در ارتباط با درگاه برای تأیید پرداخت',
                    'data' => null
                ];
            }

        } catch (\Exception $e) {
            Log::error('AsanPardakht Verify Exception', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return [
                'status' => 500,
                'message' => 'خطا در تأیید پرداخت',
                'data' => null
            ];
        }
    }

    /**
     * ریفاند پرداخت
     *
     * @param string $refNum
     * @param int $amount
     * @return array
     */
    public function refundPayment(string $refNum, int $amount): array
    {
        try {
            $payload = [
                'merchantConfigurationId' => $this->merchantId,
                'refNum' => $refNum,
                'amountInRials' => $amount
            ];

            Log::info('AsanPardakht Refund Request', ['payload' => $payload]);

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])->post($this->baseUrl . '/Refund', $payload);

            if ($response->successful()) {
                $result = $response->json();

                if (isset($result['status']) && $result['status'] == 0) {
                    return [
                        'status' => 200,
                        'message' => 'ریفاند با موفقیت انجام شد',
                        'data' => $result
                    ];
                } else {
                    Log::error('AsanPardakht Refund Error', ['response' => $result]);
                    return [
                        'status' => 500,
                        'message' => 'ریفاند ناموفق: ' . ($result['message'] ?? 'خطای نامشخص'),
                        'data' => $result
                    ];
                }
            } else {
                Log::error('AsanPardakht Refund HTTP Error', ['status' => $response->status(), 'body' => $response->body()]);
                return [
                    'status' => 500,
                    'message' => 'خطا در ارتباط با درگاه برای ریفاند',
                    'data' => null
                ];
            }

        } catch (\Exception $e) {
            Log::error('AsanPardakht Refund Exception', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return [
                'status' => 500,
                'message' => 'خطا در ریفاند',
                'data' => null
            ];
        }
    }
}
