<?php

namespace App\Repositories\Payment\PaymentGateways;


use App\Models\ChargeWallet;
use App\Repositories\Order\OrderRepository;
use App\Repositories\Payment\PaymentGateways\PaymentGatewayInterface;
use App\Repositories\Setting\SettingRepository;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class BitPayGateway implements PaymentGatewayInterface
{
    public function setPay($paymentCode, $amount, $callbackRoute = null, $orderId = null): array
    {
        if ($callbackRoute == null)
            $callbackRoute = route('payConfirm', ['gateway_id' => 1]);
// Retrieve MerchantID
        try {
            $merchantSetting = app(SettingRepository::class)->getSettingWithName('bitPay');
            $MerchantID = $merchantSetting['value'];
            Log::info("MerchantID retrieved: " . $MerchantID);
        } catch (\Exception $e) {
            Log::error("Error retrieving MerchantID: " . $e->getMessage());
            return ['status' => 500, 'message' => 'خطا در دریافت شناسه پذیرنده', 'data' => null];
        }

        $payload = [
            'api' => $MerchantID,
            'redirect' => $callbackRoute,
            'amount' => (int)$amount * 10,
        ];

        Log::info("Payload for HTTP POST: ", $payload);

        try {

            $url = 'https://bitpay.ir/payment/gateway-send';
            $client = new Client();
            $response = $client->post($url, [
                'form_params' => $payload
            ]);
            $result = (string)$response->getBody();
            Log::info("Response from bitPay: " . json_encode($result));
        } catch (\Exception $e) {
            Log::error("HTTP request failed: " . $e->getMessage());
            return ['status' => 500, 'message' => 'خطا در ارتباط با درگاه پرداخت', 'data' => null];
        }

        if ($result > 0 && is_numeric($result)) {
            Log::info("Valid response received: " . $result);
            return ['status' => 200, 'message' => 'با موفقیت درخواست ثبت شد.', 'data' => ['route' => "https://bitpay.ir/payment/gateway-$result-get", 'codeGateway' => $result]];
        } else {
            Log::error("Invalid response: " . json_encode($result));
            return ['status' => 500, 'message' => 'خطا در ارتباط با درگاه پرداخت', 'data' => null];
        }
    }


    public function confirmPay($data): array
    {
        $merchantSetting = app(SettingRepository::class)->getSettingWithName('bitPay');
        $MerchantID = $merchantSetting['value'];
        $trans_id = $data['trans_id'] ?? 0;
        $id_get = $data['id_get'] ?? 0;
        $order = app(OrderRepository::class)->getByCodeGateway($id_get);
        if ($order == null) {
            return ['status' => 404, 'message' => 'سفارشی یافت نشد', 'data' => []];
        }
        $amount = $order->order_price - $order->wallet_use_price - $order->cash_back_price;
        try {
            $payload = [
                'api' => "$MerchantID",
                'trans_id' => (int)$trans_id,
                'id_get' => (int)$id_get,
                'amount' => $amount * 10,
            ];
            $url = 'https://bitpay.ir/payment/gateway-result-second';
            $client = new Client();
            $response = $client->post($url,
                ['form_params' => $payload
                ]
            );
            $result = (string)$response->getBody();
            Log::info("Response from bitPay: " . json_encode($result));
        } catch (\Exception $e) {
            Log::error("HTTP request failed: " . $e->getMessage());
            return ['status' => 500, 'message' => 'عملیات پرداخت با مشکل موجه شد.', 'data' => ['order' => $order ?? []]];
        }


        if ((int)$result == 1 || (int)$result == 11) {
            return ['status' => 200, 'message' => '', 'data' => ['order' => $order, 'refId' => $trans_id]];
        }
        return ['status' => 500, 'message' => 'عملیات پرداخت با مشکل موجه شد.', 'data' => ['order' => $order ?? []]];
    }

    public function confirmChargePay($data): array
    {
        $merchantSetting = app(SettingRepository::class)->getSettingWithName('bitPay');
        $MerchantID = $merchantSetting['value'];
        $trans_id = $data['trans_id'] ?? 0;
        $id_get = $data['id_get'] ?? 0;
        $order = ChargeWallet::where('payment_code_gateway', $id_get)->first();
        if ($order == null) {
            return ['status' => 404, 'message' => 'سفارشی یافت نشد', 'data' => []];
        }
        $amount = $order->amount;
        try {
            $payload = [
                'api' => "$MerchantID",
                'trans_id' => (int)$trans_id,
                'id_get' => (int)$id_get,
                'amount' => $amount * 10,
            ];
            $url = 'https://bitpay.ir/payment/gateway-result-second';
            $client = new Client();
            $response = $client->post($url,
                ['form_params' => $payload]
            );
            $result = (string)$response->getBody();
            Log::info("Response from bitPay: " . json_encode($result));
        } catch (\Exception $e) {
            Log::error("HTTP request failed: " . $e->getMessage());
            return ['status' => 500, 'message' => 'عملیات پرداخت با مشکل موجه شد.', 'data' => ['order' => $order ?? []]];
        }

        if ((int)$result == 1 || (int)$result == 11) {
            return ['status' => 200, 'message' => '', 'data' => ['order' => $order, 'refId' => $trans_id]];
        }
        return ['status' => 500, 'message' => 'عملیات پرداخت با مشکل موجه شد.', 'data' => ['order' => $order ?? []]];
    }
}
