<?php

namespace App\Repositories\Payment\PaymentGateways;

use App\Models\ChargeWallet;
use App\Repositories\Order\OrderRepository;
use Illuminate\Support\Facades\Log;

class MellatGateway implements PaymentGatewayInterface
{
    protected string $terminalId;
    protected string $userName;
    protected string $userPassword;
    protected string $wsdl;

    public function __construct()
    {
        $this->terminalId = env('MELLAT_TERMINAL_ID');
        $this->userName = env('MELLAT_USERNAME');
        $this->userPassword = env('MELLAT_PASSWORD');
        $this->wsdl = 'https://bpm.shaparak.ir/pgwchannel/services/pgw?wsdl';
    }

    public function setPay($paymentCode, $amount, $callbackRoute = null, $orderId = null): array
    {
        if ($callbackRoute == null)
            $callbackRoute = route('payConfirm', ['gateway_id' => 6]); // ID اختصاصی ملت

        $params = [
            'terminalId' => $this->terminalId,
            'userName' => $this->userName,
            'userPassword' => $this->userPassword,
            'orderId' => $paymentCode,
            'amount' => $amount,
            'localDate' => now()->format('Ymd'),
            'localTime' => now()->format('His'),
            'additionalData' => '',
            'callBackUrl' => $callbackRoute,
            'payerId' => 0,
        ];

        try {
            $client = new \SoapClient($this->wsdl);
            $response = $client->bpPayRequest($params);
            $result = explode(',', $response->return);

            if ($result[0] == "0") {
                return [
                    'status' => 200,
                    'message' => 'درخواست پرداخت با موفقیت انجام شد.',
                    'data' => [
                        'route' => "https://bpm.shaparak.ir/pgwchannel/startpay.mellat?RefId=" . $result[1],
                        'codeGateway' => $result[1]
                    ]
                ];
            }

            return [
                'status' => 500,
                'message' => 'خطا در دریافت RefId از بانک ملت. کد خطا: ' . $result[0],
                'data' => null
            ];
        } catch (\Throwable $e) {
            Log::error('Mellat Pay Request Exception', ['msg' => $e->getMessage()]);
            return [
                'status' => 500,
                'message' => 'خطا در اتصال به بانک ملت',
                'data' => null
            ];
        }
    }

    public function confirmPay($data): array
    {
        // اصلاح: به‌جای SaleReferenceId، از SaleOrderId برای پیدا کردن سفارش استفاده می‌کنیم
        // چون SaleOrderId همون payment_code توی جدول orders است
        $order = app(OrderRepository::class)->getByPaymentCode($data['SaleOrderId']);
        if (!$order) {
            return [
                'status' => 404,
                'message' => 'سفارش یافت نشد.',
                'data' => ['order' => null]
            ];
        }

        if ($data['ResCode'] != 0) {
            return [
                'status' => 500,
                'message' => 'پرداخت توسط کاربر لغو شد یا ناموفق بود.',
                'data' => ['order' => $order]
            ];
        }

        $verified = $this->verify(
            $order->payment_code,
            $data['SaleOrderId'],
            $data['SaleReferenceId']
        );

        if ($verified) {
            return [
                'status' => 200,
                'message' => '',
                'data' => [
                    'order' => $order,
                    'refId' => $data['SaleReferenceId']
                ]
            ];
        }

        return [
            'status' => 500,
            'message' => 'تأیید پرداخت ناموفق بود.',
            'data' => ['order' => $order]
        ];
    }

    public function confirmChargePay($data): array
    {
        $order = ChargeWallet::where('payment_code_gateway', $data['SaleReferenceId'])->first();

        if (!$order) {
            return [
                'status' => 404,
                'message' => 'تراکنش کیف پول یافت نشد.',
                'data' => ['order' => null]
            ];
        }

        if ($data['ResCode'] != 0) {
            return [
                'status' => 500,
                'message' => 'پرداخت توسط کاربر لغو شد یا ناموفق بود.',
                'data' => ['order' => $order]
            ];
        }

        $verified = $this->verify(
            $order->payment_code,
            $data['SaleOrderId'],
            $data['SaleReferenceId']
        );

        if ($verified) {
            return [
                'status' => 200,
                'message' => '',
                'data' => [
                    'order' => $order,
                    'refId' => $data['SaleReferenceId']
                ]
            ];
        }

        return [
            'status' => 500,
            'message' => 'تأیید پرداخت کیف پول ناموفق بود.',
            'data' => ['order' => $order]
        ];
    }

    protected function verify(string $orderId, string $saleOrderId, string $saleReferenceId): bool
    {
        $params = [
            'terminalId' => $this->terminalId,
            'userName' => $this->userName,
            'userPassword' => $this->userPassword,
            'orderId' => $orderId,
            'saleOrderId' => $saleOrderId,
            'saleReferenceId' => $saleReferenceId,
        ];

        try {
            $client = new \SoapClient($this->wsdl);
            $result = $client->bpVerifyRequest($params);
            return $result->return == "0";
        } catch (\Throwable $e) {
            Log::error('Mellat Verify Exception', ['msg' => $e->getMessage()]);
            return false;
        }
    }
}
