<?php

namespace App\Repositories\PopupAlarm;

use App\Models\PopupAlarm;
use App\Models\PopupAlarmView;
use App\Repositories\BaseRepository\BaseRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Log;

class EloquentRepository extends BaseRepository implements RepositoryInterface
{
    public function __construct(PopupAlarm $model)
    {
        parent::__construct($model);
    }

    /**
     * ایجاد آلارم جدید
     */
    public function create(array $data): PopupAlarm
    {
        return $this->model->create($data);
    }

    /**
     * دریافت آلارم‌های فعال رستوران
     */
    public function getActiveAlarmsForBranch($branchId)
    {
        return $this->model->where('branch_id', $branchId)->where('status', 'active')->where('start_time', '<=', now())->where('end_time', '>=', now())->orderBy('created_at', 'desc')->get();
    }

    /**
     * دریافت آلارم‌های فعال که کاربر ندیده
     */
    public function getUnseenActiveAlarmsForUser($userId, $branchId = null)
    {
        $query = $this->model->where('status', 'active')->where('start_time', '<=', Carbon::now())->where('end_time', '>=',Carbon::now());

        if ($userId != null) {
            $query->whereDoesntHave('views', function ($q) use ($userId) {
                $q->where('user_id', $userId);
            });
        }

        if ($branchId) {
            $query->where('branch_id', $branchId);
        }
        $data = $query->orderBy('created_at', 'desc')->get();
        return $data;
    }

    /**
     * ثبت مشاهده آلارم توسط کاربر
     */
    public function markAsViewed($alarmId, $userId)
    {
        return PopupAlarmView::create([
            'popup_alarm_id' => $alarmId,
            'user_id' => $userId,
            'viewed_at' => now(),
        ]);
    }

    /**
     * بررسی اینکه کاربر آلارم را دیده یا نه
     */
    public function hasUserSeenAlarm($alarmId, $userId)
    {
        return PopupAlarmView::where('popup_alarm_id', $alarmId)->where('user_id', $userId)->exists();
    }

    /**
     * دریافت لیست آلارم‌های رستوران
     */
    public function getBranchAlarms($branchId, $perPage = 15)
    {
        return $this->model
            ->where('branch_id', $branchId)
            ->with([
                'views' => function ($query) {
                    $query->select('popup_alarm_id', 'user_id', 'viewed_at');
                },
            ])
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);
    }

    /**
     * به‌روزرسانی آلارم
     */
    public function update($id, array $data)
    {
        $alarm = $this->model->findOrFail($id);
        return $alarm->update($data);
    }

    /**
     * حذف آلارم
     */
    public function delete($id)
    {
        $alarm = $this->model->findOrFail($id);
        return $alarm->delete();
    }

    /**
     * دریافت آمار مشاهده آلارم
     */
    public function getAlarmViewStats($alarmId)
    {
        $alarm = $this->model->findOrFail($alarmId);

        $totalViews = $alarm->views()->count();
        $uniqueUsers = $alarm->views()->distinct('user_id')->count();

        return [
            'total_views' => $totalViews,
            'unique_users' => $uniqueUsers,
            'views_today' => $alarm->views()->whereDate('viewed_at', today())->count(),
            'views_this_week' => $alarm
                ->views()
                ->whereBetween('viewed_at', [now()->startOfWeek(), now()->endOfWeek()])
                ->count(),
        ];
    }

    public function getAlarmById($id)
    {
        $alarm = $this->model->findOrFail($id);
        return $alarm;
    }
}
