<?php

namespace App\Repositories\Print;

use App\Models\Order;
use App\Models\PrintJob;
use App\Models\PrinterDevice;
use App\Repositories\BaseRepository\BaseRepository;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class EloquentRepository extends BaseRepository{

    public function __construct(PrinterDevice $model)
    {
        parent::__construct($model);
    }


    public function getWithFilter(array $filter = [])
    {
        try {
            $data = $this->model->where(function ($q) use ($filter) {
                if (isset($filter['search']) && !empty($filter['search'])) {
                    $q->where('name', 'like', '%' . $filter['search'] . '%')->orWhere('family', 'like', '%' . $filter['search']);
                }
            })->where(function ($q) use ($filter) {
                if (isset($filter['mobile']) && !empty($filter['mobile'])) {
                    $q->where('mobile', $filter['mobile']);
                }
            })->where(function ($q) use ($filter) {
                if (isset($filter['area_id']) && !empty($filter['area_id'])) {
                    $q->where('area_id', $filter['area_id']);
                }
            });
            if (isset($filter['export']) and $filter['export'] == true) {
                $export = clone $data;
                app(ExportImportRepository::class)->createExcelFileUsers($export->get()->toArray());
            }
            if (isset($filter['paginate']) and $filter['paginate'])
                return $data->paginate(25);
            return $data->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }




    public function eligiblePrintersForOrder(Order $order): Collection
    {
        return PrinterDevice::query()
            ->where('is_active', true)
            ->when($order->store_id ?? null, fn($q, $storeId) => $q->where(function($q) use ($storeId) {
                $q->whereNull('store_id')->orWhere('store_id', $storeId);
            }))
            ->get()
            ->filter(fn(PrinterDevice $p) => $this->orderPassesFilters($p, $order) && $this->isWithinTimeWindow($p));
    }

    public function shouldPrintItem(PrinterDevice $printer, $item): bool
    {
        $f = $printer->item_filters ?: [];
        $inc = $f['include'] ?? [];
        $exc = $f['exclude'] ?? [];
        $includeAll = (bool)($f['include_all'] ?? false);

        $productId  = $item->product_id ?? null;
        $categoryId = $item->category_id ?? null;
        $sku        = (string)($item->sku ?? '');
        $tags       = $item->tags ?? [];
        if (is_string($tags)) {
            $tags = array_filter(array_map('trim', explode(',', $tags)));
        }

        // Exclude
        if (!empty($exc['product_ids']) && in_array($productId, $exc['product_ids'])) return false;
        if (!empty($exc['tags']) && array_intersect($exc['tags'], $tags)) return false;

        if ($includeAll) return true;

        // Include
        $match = false;
        if (!empty($inc['product_ids']) && in_array($productId, $inc['product_ids'])) $match = true;
        if (!empty($inc['category_ids']) && in_array($categoryId, $inc['category_ids'])) $match = true;
        if (!empty($inc['sku_prefixes']) && $sku) {
            foreach ($inc['sku_prefixes'] as $pfx) {
                if (str_starts_with($sku, $pfx)) { $match = true; break; }
            }
        }
        if (!empty($inc['tags']) && array_intersect($inc['tags'], $tags)) $match = true;

        return $match;
    }

    public function mapOrderItemsToPrinters(Order $order): array
    {
        $printers = $this->eligiblePrintersForOrder($order)->sortByDesc('priority');
        $map = [];
        foreach ($printers as $printer) {
            foreach ($order->items as $item) {
                if ($this->shouldPrintItem($printer, $item)) {
                    $map[$printer->id][] = $item;
                }
            }
        }
        // فقط پرینترهایی که حداقل یک آیتم دارند
        return array_filter($map);
    }

    public function createJobsForOrder(Order $order, array $map, ?string $payloadPath = null): Collection
    {
        $jobs = collect();
        foreach ($map as $printerId => $items) {
            $jobs->push(PrintJob::create([
                'order_id'          => $order->id,
                'printer_device_id' => $printerId,
                'status'            => 'queued',
                'attempts'          => 0,
                'payload_path'      => $payloadPath, // می‌تونی برای هر پرینتر فایل جدا بسازی
            ]));
        }
        return $jobs;
    }

    public function markJob(PrintJob $job, string $status, ?string $error = null): void
    {
        $job->status = $status;
        if ($error) {
            $job->last_error = $error;
            $job->increment('attempts');
        }
        $job->save();

        // به‌روز کردن اطلاعات پرینتر
        $job->printer?->forceFill([
            'last_job_at'   => now(),
            'last_error_at' => $error ? now() : $job->printer->last_error_at,
            'last_error'    => $error ?? $job->printer->last_error,
        ])->save();
    }

    private function orderPassesFilters(PrinterDevice $printer, Order $order): bool
    {
        $f = $printer->order_filters ?: [];
        if (empty($f)) return true;

        if (!empty($f['statuses']) && !in_array($order->status, $f['statuses'])) return false;
        if (isset($f['min_total']) && (float)$order->total < (float)$f['min_total']) return false;
        if (!empty($f['payment_methods']) && !in_array($order->payment_method ?? '', $f['payment_methods'])) return false;

        return true;
    }

    private function isWithinTimeWindow(PrinterDevice $printer): bool
    {
        $tw = $printer->enabled_time_windows ?? null;
        if (!$tw) return true;

        $now = Carbon::now();
        $day = (int)$now->isoWeekday(); // 1..7
        if (!empty($tw['days']) && !in_array($day, $tw['days'])) return false;

        if (!empty($tw['from']) && !empty($tw['to'])) {
            $from = Carbon::parse($tw['from']);
            $to   = Carbon::parse($tw['to']);
            if (!$now->between($from, $to, true)) return false;
        }
        return true;
    }
}
