<?php

namespace App\Repositories\PrinterSetting;

use App\Models\PrinterSetting;
use App\Repositories\BaseRepository\BaseRepository;
use Illuminate\Support\Facades\Log;

abstract class EloquentRepository extends BaseRepository
{
    public function __construct(PrinterSetting $model)
    {
        parent::__construct($model);
    }

    public function getWithFilter(array $filter = [], $paginate = false)
    {
        try {
            $data = $this->model->with(['branch'])->where(function ($q) use ($filter) {
                if (isset($filter['search']) && !empty($filter['search'])) {
                    $q->where('name', 'like', '%' . $filter['search'] . '%')
                      ->orWhere('title', 'like', '%' . $filter['search'] . '%');
                }

                if (isset($filter['branch_id']) && !empty($filter['branch_id'])) {
                    $q->where('branch_id', $filter['branch_id']);
                }

                if (isset($filter['status']) && !empty($filter['status'])) {
                    $q->where('status', $filter['status']);
                }

                if (isset($filter['type']) && !empty($filter['type'])) {
                    $q->where('type', $filter['type']);
                }
            });

            if ($paginate) {
                return $data->orderBy('id', 'desc')->paginate(25);
            } else {
                return $data->orderBy('id', 'desc')->get();
            }
        } catch (\Exception $e) {
            Log::error('PrinterSetting getWithFilter error: ' . $e->getMessage());
            return null;
        }
    }

    public function show($id, $filter = [], $with = null, $select = null)
    {
        try {
            $query = $this->model->with(['branch'])->where('id', $id);

            if (!empty($filter)) {
                $query->where($filter);
            }

            if ($with) {
                $query->with($with);
            }

            if ($select) {
                $query->select($select);
            }

            return $query->first();
        } catch (\Exception $e) {
            Log::error('PrinterSetting show error: ' . $e->getMessage());
            return null;
        }
    }

    public function getByBranch($branchId)
    {
        try {
            return $this->model->where('branch_id', $branchId)
                              ->where('status', 'active')
                              ->orderBy('id', 'desc')
                              ->get();
        } catch (\Exception $e) {
            Log::error('PrinterSetting getByBranch error: ' . $e->getMessage());
            return collect();
        }
    }

    public function getByType($type, $branchId = null)
    {
        try {
            $query = $this->model->where('type', $type);

            if ($branchId) {
                $query->where('branch_id', $branchId);
            }

            return $query->where('status', 'active')
                        ->orderBy('id', 'desc')
                        ->get();
        } catch (\Exception $e) {
            Log::error('PrinterSetting getByType error: ' . $e->getMessage());
            return collect();
        }
    }
}
