<?php

namespace App\Repositories\Product;

use App\Enums\BranchStatus;
use App\Enums\ProductStatus;
use App\Models\Product;
use App\Models\ProductDefault;
use App\Models\Variety;
use App\Repositories\BaseRepository\BaseRepository;
use App\Repositories\Category\CategoryRepository;
use App\Repositories\ExportImport\ExportImportRepository;
use Illuminate\Support\Facades\Log;

abstract class EloquentRepository extends BaseRepository
{
    public function __construct(Product $model)
    {
        parent::__construct($model);
    }

    public function getWithFilter(array $filter = [], $paginate = false)
    {
        try {
            Log::info("Csdcds");
            $data = $this->model->where(function ($q) use ($filter) {
                if (isset($filter['search']) && !empty($filter['search'])) {
                    $q->where('title', 'like', '%' . $filter['search'] . '%');
                }
                if (isset($filter['category_id']) && !empty($filter['category_id'])) {
                    $categoryChild = app(CategoryRepository::class)->getAllChildArray($filter['category_id']);
                    $q->whereHas('category', function ($query) use ($categoryChild) {
                        $query->whereIn('id', $categoryChild);
                    });
                }

                if (isset($filter['branch_id']) && !empty($filter['branch_id'])) {
                    $q->where('branch_id', $filter['branch_id']);
                }
                if (isset($filter['status']) && !empty($filter['status'])) {
                    $q->where('status', $filter['status']);
                }
                if (isset($filter['agent_id']) && !empty($filter['agent_id'])) {

                    $q->whereHas('branch', function ($query) use ($filter) {
                        $query->where('agent_id', $filter['agent_id']);
                    });
                }


            })->with('branch', 'varieties', 'category', 'material')->orderBy('id', 'desc');

            if (isset($filter['export']) and $filter['export'] == true) {
                $export = clone $data;
                app(ExportImportRepository::class)->createExcelFileProducts($export->get()->toArray());
            }

            if ($paginate)
                return $data->paginate(25);
            return $data->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }


    public function getByFilterFront($request)
    {
        // تنظیم مقدار پیش‌فرض برای ترتیب
        $field = "id";
        $type = "DESC";

        if (!empty($request['order_by'])) {
            switch ($request['order_by']) {
                case "rate":
                    $field = "rate";
                    $type = "DESC";
                    break;
                case "desc":
                    $field = "id";
                    $type = "DESC";
                    break;
            }
        }

        return Product::where('status', ProductStatus::ACTIVE)
            ->when(!empty($request['branch_id']), function ($query) use ($request) {
                $query->where('branch_id', $request['branch_id']);
            })
            ->when(!empty($request['category_id']), function ($query) use ($request) {
                $categoryChild = app(CategoryRepository::class)->getAllChildArray((int)$request['category_id']);
                $query->whereHas('category', fn($q) => $q->whereIn('id', $categoryChild));
            })
            ->when(!empty($request['title']), function ($query) use ($request) {
                $query->where('title', 'LIKE', '%' . trim($request['title']) . '%');
            })
            ->when(!empty($request['material_ids']), function ($query) use ($request) {
                $materials = explode('-', $request['material_ids']) ?? [];
                $query->whereHas('material', fn($q) => $q->whereIn('materials.id', $materials));
            })
            ->whereHas('variety', function ($query) use ($request) {
                if (!empty($request['has_discount'])) {
                    $query->where('discount', '>', 0)->orderByDesc('discount');
                }
                if (!empty($request['price_start'])) {
                    $query->where('price', '>=', $request['price_start']);
                }
                if (!empty($request['price_end'])) {
                    $query->where('price', '<=', $request['price_end']);
                }
            })
            ->whereHas('branch', function ($query) use ($request) {
                if (!empty($request['lat']) && !empty($request['lng'])) {
                    $lat = $request['lat'];
                    $lng = $request['lng'];
                    $radius = 10;
                    $haversine = "(6371 * acos(cos(radians($lat)) * cos(radians(lat)) * cos(radians(lng) - radians($lng)) + sin(radians($lat)) * sin(radians(lat))))";
                    $query->selectRaw("{$haversine} AS distance")
                        ->whereRaw("{$haversine} < ?", [$radius]);
                }
                $query->where("status", BranchStatus::ACTIVE->value);
            })
            ->with([
                'varieties' => function ($query) {
                    $query->orderByDesc('discount');
                },
                'extra:id,title,price',
                'material:id,title,unit_id,image,price_per_unit',
                'branch:id,title,address,lat,lng,vendor_id,min_order,sm_description,logo,description'
            ])
            ->orderBy($field, $type)
            ->paginate($request->paginate ?? 10);
    }

    public function search($filter, $paginate = false, $selects = null)
    {
        try {
            $data = $this->model->where(function ($q) use ($filter) {
                if (isset($filter['search']) && !empty($filter['search'])) {
                    $q->where('title', 'like', '%' . $filter['search'] . '%');
                }
            })->with('branch:id,title', 'varieties');
            if ($selects != null)
                $data->select($selects);
            if ($paginate)
                return $data->paginate(25);
            return $data->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }


    public function storeP($data)
    {
        try {
            $n = new Product();
            $n->title = $data['title'];
            $n->branch_id = $data['branch_id'];
            $n->menu_id = $data['menu_id'];
            $n->category_id = $data['category_id'];
            $n->image = $data['image'];
            $n->description = $data['description'];
            $n->save();
            return $n;
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }

    public function storeWithIdsDefault($data)
    {
        try {
            $productDefaults = ProductDefault::whereIn('id', $data['ids'])->get();
            foreach ($productDefaults as $item) {
                $n = new Product();
                $n->branch_id = $data['branch_id'];
                $n->title = $item['title'];
                $n->description = $item['description'];
                $n->min_description = $item['sm_description'];
                $n->image = $item['image'];
                $n->images = $item['images'];
                $n->category_id = $item['category_id'];
                $n->menu_id = 0;
                $n->product_defualt_id = $item->id;
                $n->save();
            }
            return true;
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }


    public function showWithRel($id)
    {
        try {
            return Product::where('id', $id)->with('branch', 'varieties', 'category', 'files', 'material')->first();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }

    public function updateCountVariety($data)
    {
        try {
            Variety::find($data['variety_id'])->update($data);
            return ['data' => [], 'message' => null, 'status' => 200];
        } catch (\Exception $e) {
            return ['data' => [], 'message' => $e->getMessage(), 'status' => 500];
        }
    }

    public function updateOrStoreVarieties($data, $productId)
    {
        try {

            $productVarieties = Variety::where('product_id', $productId)->get()->pluck('title', 'id')->toArray();
            Log::info($productVarieties);
            foreach ($data as $varietyData) {
                $varietyId = $varietyData['id'] ?? null;
                Log::info($varietyId);

                unset($productVarieties[$varietyId]);
                $varietyData['product_id'] = $productId;
                $variety = Variety::updateOrCreate(
                    ['id' => $varietyId],
                    $varietyData
                );
            }
            Log::info($productVarieties);
            Variety::whereIn('id', array_keys($productVarieties))->delete();
            return ['status' => '200', 'data' => [], 'message' => null];
        } catch (\Exception $e) {
            return ['status' => '500', 'data' => [], 'message' => $e->getMessage()];
        }
    }


    public function getProductMaterial($id)
    {
        return Product::where(['id' => $id])->with('material:id,title,unit_id,image,price_per_unit,calorie_per_unit,carbohydrate_per_unit,fat_per_unit,protein_per_unit')->first();
    }

    public function getProductWithAllRel($id)
    {
        return Product::where(['id' => $id])->where('status', ProductStatus::ACTIVE)->with('varieties', 'extra:id,title,price', 'material:id,title,unit_id,image,price_per_unit', 'branch:id,title,address,lat,lng,vendor_id,min_order,sm_description,logo,description')->first();
    }


    public function getAllCalories($id)
    {
        $product = $this->getProductMaterial($id);
        $protein = 0;
        $carbohydrate = 0;
        $fat = 0;
        foreach ($product->material as $item) {
            if ($item->pivot->usage_per_unit != null) {
                $protein += $item->pivot->usage_per_unit * $item->protein_per_unit;
                $carbohydrate += $item->pivot->usage_per_unit * $item->carbohydrate_per_unit;
                $fat += $item->pivot->usage_per_unit * $item->fat_per_unit;
            }
        }
        $energy = $protein + $fat + $carbohydrate;
        return ['data' => ['protein' => [$protein, number_format($this->calculatePercentage($protein, $energy), 2)],
            'carbohydrate' => [$carbohydrate, number_format($this->calculatePercentage($carbohydrate, $energy), 2)],
            'fat' => [$fat, number_format($this->calculatePercentage($fat, $energy), 2)], 'energy' => $fat + $carbohydrate + $protein]];
    }

    public function calculatePercentage($part, $whole)
    {
        try {
            if ($whole != 0 and $part != 0)
                return ($part / $whole) * 100;
            return 0;
        } catch (\Exception $e) {
            return 0;
        }
    }


    public function sitemapListProducts()
    {
        try {
            return $this->model->select('id', 'title')->get()->toArray();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }
}
