<?php

namespace App\Repositories\Settlement;

use App\Enums\SettlementRequestStatus;
use App\Enums\TransactionType;
use App\Models\Branch;
use App\Models\SettlementRequest;
use App\Models\User;
use App\Repositories\BaseRepository\BaseRepository;
use App\Repositories\Transaction\TransactionRepository;
use App\Repositories\User\UserRepository;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

abstract class EloquentRepository extends BaseRepository
{
    public function __construct(SettlementRequest $model)
    {
        parent::__construct($model);
    }

    public function getWithFilter(array $filter = [], $paginate = false)
    {
        try {
            $data = $this->model->where(function ($q) use ($filter) {
                if (isset($filter['status']) && !empty($filter['status'])) {
                    $q->where('status', $filter['status']);
                }

                if (isset($filter['branch_id']) && !empty($filter['branch_id'])) {
                    $q->where('branch_id', $filter['branch_id']);
                }
            })->with('branch')->orderBy('id', 'DESC');
            if ($paginate)
                return $data->paginate(25);
            return $data->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }

    public function getWalletRestaurant(array $filter = [], $paginate = false)
    {
        try {
            $restaurant = User::where('role_id', 3)->whereHas('transactions', function ($query) {
                $query->select(DB::raw('SUM(amount) as total_amount'))
                    ->groupBy('user_id')
                    ->havingRaw('total_amount > 0');
            })->with(['branch:id,title', 'transactions' => function ($query) {
                $query->select('user_id', DB::raw('SUM(amount) as total_amount'))
                    ->groupBy('user_id')
                    ->havingRaw('total_amount > 0');
            }]);
            if ($paginate)
                return $restaurant->paginate(25);
            return $restaurant->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }


    public function changeStatus($settlementId, $status)
    {
        try {
            if ($status === SettlementRequestStatus::SUCCESS->value) {
                $s = $this->model->where("id", $settlementId)->first();
                $branch = Branch::find($s->branch_id);

                $user = User::where("role_id", 2)->where('target_role_id', $branch->id)->first();
                if ($user == null) {
                    return false;
                }
                $wallet = app(TransactionRepository::class)->userInventoryWithUser($user->id);
                if ($wallet > $s->amount) {
                    app(TransactionRepository::class)->storeP(['amount' => $s->amount * -1, 'user_id' => $user->id, 'transaction_type' => TransactionType::CHARGE, 'target_id' => 0,
                        'description' => 'تسویه شد.']);
                    $s->update(['status' => $status]);
                } else {
                    return false;
                }
            } else {
                $this->model->where("id", $settlementId)->update(["status" => $status]);
            }
            return true;
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }

}
