<?php

namespace App\Repositories\User;

use App\Enums\OrderStatus;
use App\Models\Order;
use App\Models\Transaction;
use App\Models\UserBranchWallet;
use App\Models\ValidationCode;
use App\Repositories\BaseRepository\BaseRepository;
use App\Repositories\ExportImport\ExportImportRepository;
use App\Repositories\SMS\SMSRepositoryInterface;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

abstract class  EloquentRepository extends BaseRepository
{
    protected $SMSRepository;

    public function __construct(User $model, SMSRepositoryInterface $SMSRepository)
    {
        parent::__construct($model);
        $this->SMSRepository = $SMSRepository;
    }


    public function getWithFilter(array $filter = [], $paginate = false)
    {
        try {
            $data = $this->model->where(function ($q) use ($filter) {
                if (isset($filter['search']) && !empty($filter['search'])) {
                    $q->where('name', 'like', '%' . $filter['search'] . '%')->orWhere('family', 'like', '%' . $filter['search'])->orWhere('mobile', 'like', '%' . $filter['search']);
                }
            })->where(function ($q) use ($filter) {
                if (isset($filter['role_id']) && !empty($filter['role_id'])) {
                    $q->where('role_id', $filter['role_id']);
                }
                if (isset($filter['mobile']) && !empty($filter['mobile'])) {
                    $q->where('mobile', $filter['mobile']);
                }
            });
            if (isset($filter['export']) and $filter['export'] == true) {
                $export = clone $data;
                app(ExportImportRepository::class)->createExcelFileUsers($export->get()->toArray());
            }
            if ($paginate)
                return $data->orderBy('id','desc')->paginate(25);
            return $data->get();
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getDriversMobileNumber()
    {
        return $this->model->where('role_id', 5)->select('mobile')->get()->pluck('mobile')->toArray();
    }

    public function storeP($data)
    {
        try {
            $n = new User();
            $n->name = $data['name'];
            $n->family = $data['family'];
            $n->mobile = $data['mobile'];
            $n->email = $data['email'] ?? null;
            $n->birth_date = $data['birth_date'] ?? null;
            $n->username = $data['username'] ?? null;
            $n->tel = $data['tel'] ?? null;
            $n->role_id = $data['role_id'] ?? 4;
            $n->target_role_id = $data['target_role_id'] ?? 0;
            $n->referral_code = Str::random(5);
            $n->save();
            return $n;
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }


    public function checkLatestSmsCode($mobile)
    {
        $timeGapSendCode = 2;
        $dateTime = Carbon::now()->subMinutes($timeGapSendCode);
        return ValidationCode::where('created_at', '>=', $dateTime)->where('mobile', $mobile)->where('used_at', null)->orderBy('id', 'desc')->first();

    }


    public function sendSmsCode($mobile)
    {
        $code = rand(1000, 9999);
        if ($mobile == "09025045706")
            $code = 1111;
        $code = 1111;

        $this->saveValidationCode($mobile, $code);
        $this->SMSRepository->sendSMSSendCode($code, $mobile);
        return true;
    }

    public function saveValidationCode($mobile, $code)
    {
        $vCode = new ValidationCode();
        $vCode->mobile = $mobile;
        $vCode->code = $code;
        $vCode->save();
        return true;
    }


    public function checkSmsCode($mobile, $code)
    {
        $timeGapSendCode = config('settings.timeGapSendCode') ?? 2;
        $dateTime = Carbon::now()->subMinutes($timeGapSendCode);
        $record = ValidationCode::where('created_at', '>=', $dateTime)->where('mobile', $mobile)->orderBy('id', 'desc')->first();
        if ($record and $record->code == $code) {
            $user = User::where('mobile', $mobile)->first();
            $isNewUser = false;
            if (!$user) {
                $isNewUser = true;
                $user = new User();
                $user->mobile = $mobile;
                $user->referral_code = Str::random(5);
                $user->save();
            }
            Auth::loginUsingId($user->id);
            $record->used_at = Carbon::now();
            $record->update();
            
            // Create welcome coupon for new users
            if ($isNewUser) {
                try {
                    app(\App\Services\WelcomeCouponService::class)->createWelcomeCouponForUser($user->id, $mobile);
                } catch (\Exception $e) {
                    Log::error("Failed to create welcome coupon for user {$user->id}: " . $e->getMessage());
                }
            }
            
            return "success";
        } elseif ($record and $record->code != $code)
            return "error";
        else {
            $this->sendSmsCode($mobile);
            return false;
        }

    }


    public function getProfileUser()
    {
        return User::where('id', auth()->id())->with('defaultAddress')->first();
    }

    public function getUserWithMobile($mobile)
    {
        return User::where('mobile', $mobile)->first();
    }

    public function getUserWithBranchId($branchId)
    {
        return User::where('target_role_id', $branchId)->whereIn('role_id', [3, 6])->first();
    }

    public function getOrders($filter)
    {
        return Order::where('user_id', auth()->id())->where('order_status', '!=', OrderStatus::ORDERING)
            ->where(function ($q) use ($filter) {
                if (isset($filter['status']) && !empty($filter['status'])) {
                    $q->where('order_status', $filter['status']);
                }
            })->with('branch', 'commentUser')->orderBy('id', 'desc')->get();
    }

    public function getTransactions()
    {
        return Transaction::where('user_id', auth()->id())->paginate(30);
    }

    public function getAddressUser()
    {
        return auth()->user()->address;
    }

    public function setWalletBranch($branchId, $userId, $priceCashBack, $expiredAt)
    {
        $userBranchWallet = UserBranchWallet::where('user_id', $userId)->where('branch_id', $branchId)->first();
        if ($userBranchWallet != null)
            $userBranchWallet->update(['amount' => (int)$userBranchWallet->amount + (int)$priceCashBack]);
        else {
            $new = new UserBranchWallet();
            $new->user_id = $userId;
            $new->branch_id = $branchId;
            $new->amount = $priceCashBack;
            $new->expired_at = $expiredAt;
            $new->save();
        }
    }

    public function getWalletUserInBranch($branchId, $userId)
    {
        $userBranchWallet = UserBranchWallet::where('user_id', $userId)->where('branch_id', $branchId)->first();
        if ($userBranchWallet != null)
            return $userBranchWallet->amount;
        else
            return 0;

    }

    /**
     * User Cash Back Balances
     */
    public function getUserCashBackBalances($userId)
    {
        $userBranchWallets = UserBranchWallet::where('user_id', $userId)
            ->where('amount', '>', 0)
            ->where(function($query) {
                $query->where('expired_at', '>', Carbon::now())
                      ->orWhereNull('expired_at');
            })
            ->with(['branch:id,title,logo,address'])
            ->get();

        return $userBranchWallets->map(function($wallet) {
            return [
                'branch_id' => $wallet->branch_id,
                'branch_name' => $wallet->branch->title ?? 'نامشخص',
                'branch_logo' => $wallet->branch->logo ?? null,
                'branch_address' => $wallet->branch->address ?? null,
                'balance' => $wallet->amount,
                'expired_at' => $wallet->expired_at,
                'is_expired' => $wallet->expired_at && $wallet->expired_at < Carbon::now(),
                'days_until_expiry' => $wallet->expired_at ? Carbon::now()->diffInDays($wallet->expired_at, false) : null
            ];
        });
    }

    /**
     * User Cash Back Details
     */
    public function getUserCashBackDetails($userId)
    {
        $userBranchWallets = UserBranchWallet::where('user_id', $userId)
            ->with(['branch:id,title,logo,address,cash_back_status,cash_back_percent,cash_back_max_price'])
            ->get();

        $totalBalance = 0;
        $activeBalance = 0;
        $expiredBalance = 0;

        $branches = $userBranchWallets->map(function($wallet) use (&$totalBalance, &$activeBalance, &$expiredBalance) {
            $isExpired = $wallet->expired_at && $wallet->expired_at < Carbon::now();

            if ($isExpired) {
                $expiredBalance += $wallet->amount;
            } else {
                $activeBalance += $wallet->amount;
            }

            $totalBalance += $wallet->amount;

            return [
                'branch_id' => $wallet->branch_id,
                'branch_name' => $wallet->branch->title ?? 'نامشخص',
                'branch_logo' => $wallet->branch->logo ?? null,
                'branch_address' => $wallet->branch->address ?? null,
                'balance' => $wallet->amount,
                'expired_at' => $wallet->expired_at,
                'is_expired' => $isExpired,
                'days_until_expiry' => $wallet->expired_at ? Carbon::now()->diffInDays($wallet->expired_at, false) : null,
                'cash_back_status' => $wallet->branch->cash_back_status ?? false,
                'cash_back_percent' => $wallet->branch->cash_back_percent ?? 0,
                'cash_back_max_price' => $wallet->branch->cash_back_max_price ?? 0
            ];
        });

        return [
            'summary' => [
                'total_balance' => $totalBalance,
                'active_balance' => $activeBalance,
                'expired_balance' => $expiredBalance,
                'total_branches' => $branches->count(),
                'active_branches' => $branches->where('is_expired', false)->count(),
                'expired_branches' => $branches->where('is_expired', true)->count()
            ],
            'branches' => $branches
        ];
    }

    /**
     * All Users Cash Back
     */
    public function getAllUsersCashBack($page = 1, $perPage = 20, $search = null, $branchId = null)
    {
        $query = UserBranchWallet::with(['user:id,name,family,mobile', 'branch:id,title,logo'])
            ->where('amount', '>', 0);

        // Search Filter
        if ($search) {
            $query->whereHas('user', function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('family', 'like', "%{$search}%")
                  ->orWhere('mobile', 'like', "%{$search}%");
            });
        }

        // Branch Filter
        if ($branchId) {
            $query->where('branch_id', $branchId);
        }

        $wallets = $query->paginate($perPage, ['*'], 'page', $page);

        $usersCashBack = collect($wallets->items())->groupBy('user_id')->map(function($userWallets, $userId) {
            $user = $userWallets->first()->user;
            $totalBalance = $userWallets->sum('amount');
            $activeBalance = $userWallets->where('expired_at', '>', Carbon::now())->sum('amount');
            $expiredBalance = $userWallets->where('expired_at', '<=', Carbon::now())->sum('amount');

            return [
                'user_id' => $userId,
                'user_name' => $user->name . ' ' . $user->family,
                'user_mobile' => $user->mobile,
                'total_balance' => $totalBalance,
                'active_balance' => $activeBalance,
                'expired_balance' => $expiredBalance,
                'total_branches' => $userWallets->count(),
                'active_branches' => $userWallets->where('expired_at', '>', Carbon::now())->count(),
                'expired_branches' => $userWallets->where('expired_at', '<=', Carbon::now())->count(),
                'branches' => $userWallets->map(function($wallet) {
                    return [
                        'branch_id' => $wallet->branch_id,
                        'branch_name' => $wallet->branch->title ?? 'نامشخص',
                        'branch_logo' => $wallet->branch->logo ?? null,
                        'balance' => $wallet->amount,
                        'expired_at' => $wallet->expired_at,
                        'is_expired' => $wallet->expired_at && $wallet->expired_at <= Carbon::now(),
                        'days_until_expiry' => $wallet->expired_at ? Carbon::now()->diffInDays($wallet->expired_at, false) : null
                    ];
                })
            ];
        });

        return [
            'data' => $usersCashBack->values(),
            'pagination' => [
                'current_page' => $wallets->currentPage(),
                'last_page' => $wallets->lastPage(),
                'per_page' => $wallets->perPage(),
                'total' => $wallets->total(),
                'from' => $wallets->firstItem(),
                'to' => $wallets->lastItem()
            ]
        ];
    }

    /**
     * Cash Back Statistics
     */
    public function getCashBackStatistics()
    {
        $totalWallets = UserBranchWallet::count();
        $activeWallets = UserBranchWallet::where('amount', '>', 0)
            ->where(function($query) {
                $query->where('expired_at', '>', Carbon::now())
                      ->orWhereNull('expired_at');
            })->count();
        $expiredWallets = UserBranchWallet::where('amount', '>', 0)
            ->where('expired_at', '<=', Carbon::now())->count();

        $totalBalance = UserBranchWallet::sum('amount');
        $activeBalance = UserBranchWallet::where('amount', '>', 0)
            ->where(function($query) {
                $query->where('expired_at', '>', Carbon::now())
                      ->orWhereNull('expired_at');
            })->sum('amount');
        $expiredBalance = UserBranchWallet::where('amount', '>', 0)
            ->where('expired_at', '<=', Carbon::now())->sum('amount');

        $uniqueUsers = UserBranchWallet::where('amount', '>', 0)->distinct('user_id')->count();
        $uniqueBranches = UserBranchWallet::where('amount', '>', 0)->distinct('branch_id')->count();

        // Branch Statistics
        $branchStats = UserBranchWallet::with('branch:id,title')
            ->where('amount', '>', 0)
            ->get()
            ->groupBy('branch_id')
            ->map(function($wallets, $branchId) {
                $branch = $wallets->first()->branch;
                return [
                    'branch_id' => $branchId,
                    'branch_name' => $branch->title ?? 'نامشخص',
                    'total_balance' => $wallets->sum('amount'),
                    'active_balance' => $wallets->where('expired_at', '>', Carbon::now())->sum('amount'),
                    'expired_balance' => $wallets->where('expired_at', '<=', Carbon::now())->sum('amount'),
                    'total_users' => $wallets->unique('user_id')->count(),
                    'active_users' => $wallets->where('expired_at', '>', Carbon::now())->unique('user_id')->count()
                ];
            })
            ->sortByDesc('total_balance')
            ->take(10);

        return [
            'overview' => [
                'total_wallets' => $totalWallets,
                'active_wallets' => $activeWallets,
                'expired_wallets' => $expiredWallets,
                'total_balance' => $totalBalance,
                'active_balance' => $activeBalance,
                'expired_balance' => $expiredBalance,
                'unique_users' => $uniqueUsers,
                'unique_branches' => $uniqueBranches
            ],
            'top_branches' => $branchStats->values()
        ];
    }
    public function getUserByBranchId($branchId)
    {
        return User::where('target_role_id', $branchId)->whereIn('role_id', [3, 6])->first();
    }
}
