<?php

namespace App\Repositories\UserOrder;

use App\Models\Order;
use App\Models\Favorite;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\DB;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Facades\Log;

class UserOrderRepository implements UserOrderRepositoryInterface
{
    protected Order $model;

    public function __construct(Order $model)
    {
        $this->model = $model;
    }

    /**
     * Get user's order history from a specific branch.
     */
    public function getUserBranchOrders(int $userId, int $branchId, array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        $query = $this->model->where('user_id', $userId)
            ->where('branch_id', $branchId)
            ->where('order_status', '!=', 'wating_payment')
            ->where('order_status', '!=', 'ordering');

        // Apply filters
        $this->applyFilters($query, $filters);

        // Add relationships
        $query->with([
            'user:id,name,family,mobile,email',
            'branch:id,title,address,logo,min_order,vendor_id',
            'driver.user:id,name,family,mobile',
            'deliveryType:id,title',
            'orderCart.variety:id,title,price',
            'address_order:id,title,address,lat,lng,phone_number'
        ]);

        // Order by created_at desc (newest first)
        $query->orderBy('created_at', 'desc');

        // Paginate results
        $orders = $query->paginate($perPage);

        // Add Jalali dates
        $this->addJalaliDates($orders);

        return $orders;
    }

    /**
     * Get user's order statistics from a specific branch.
     */
    public function getUserBranchOrderStats(int $userId, int $branchId, array $filters = []): array
    {
        $query = $this->model->where('user_id', $userId)
            ->where('branch_id', $branchId)
            ->where('order_status', '!=', 'wating_payment')
            ->where('order_status', '!=', 'ordering');

        // Apply filters
        $this->applyFilters($query, $filters);

        $stats = [
            'total_orders' => $query->count(),
            'delivered_orders' => (clone $query)->where('order_status', 'delivered')->count(),
            'cancelled_orders' => (clone $query)->whereIn('order_status', ['cancelled', 'returned'])->count(),
            'total_spent' => (clone $query)->where('payment_status', 'true')->sum('order_price'),
            'average_order_value' => $this->getAverageOrderValue($userId, $branchId, $filters),
            'last_order_date' => $this->getLastOrderDate($userId, $branchId),
            'favorite_products' => $this->getUserFavoriteProducts($userId, $branchId, 5),
        ];

        return $stats;
    }

    /**
     * Get user's favorite products from a specific branch.
     */
    public function getUserFavoriteProducts(int $userId, int $branchId, int $limit = 10): array
    {
        // Get products from user's order history
        $productIds = $this->model->where('user_id', $userId)
            ->where('branch_id', $branchId)
            ->where('order_status', 'delivered')
            ->join('order_carts', 'orders.id', '=', 'order_carts.order_id')
            ->join('varieties', 'order_carts.variety_id', '=', 'varieties.id')
            ->join('products', 'varieties.product_id', '=', 'products.id')
            ->select('products.id', 'products.title', 'products.image')
            ->selectRaw('SUM(order_carts.count) as total_quantity')
            ->selectRaw('COUNT(DISTINCT orders.id) as order_count')
            ->groupBy('products.id', 'products.title', 'products.image')
            ->orderBy('total_quantity', 'desc')
            ->limit($limit)
            ->get()
            ->toArray();

        return $productIds;
    }

    /**
     * Apply filters to the query.
     */
    protected function applyFilters($query, array $filters): void
    {
        if (isset($filters['order_status']) && !empty($filters['order_status'])) {
            $query->where('order_status', $filters['order_status']);
        }

        if (isset($filters['payment_status']) && !empty($filters['payment_status'])) {
            $query->where('payment_status', $filters['payment_status']);
        }

        if (isset($filters['start_date']) && isset($filters['end_date']) &&
            !empty($filters['start_date']) && !empty($filters['end_date'])) {
            $startDate = Jalalian::fromFormat('Y/m/d', $filters['start_date'])->toCarbon();
            $endDate = Jalalian::fromFormat('Y/m/d', $filters['end_date'])->toCarbon();
            $query->where('created_at', '>=', $startDate)->where('created_at', '<=', $endDate);
        }

        if (isset($filters['min_amount']) && !empty($filters['min_amount'])) {
            $query->where('order_price', '>=', $filters['min_amount']);
        }

        if (isset($filters['max_amount']) && !empty($filters['max_amount'])) {
            $query->where('order_price', '<=', $filters['max_amount']);
        }
    }

    /**
     * Add Jalali dates to orders.
     */
    protected function addJalaliDates($orders): void
    {
        foreach ($orders as $order) {
            $order->created_at_jalali = Jalalian::fromCarbon(\Carbon\Carbon::parse($order->created_at))->format('Y/m/d H:i:s');
            $order->updated_at_jalali = Jalalian::fromCarbon(\Carbon\Carbon::parse($order->updated_at))->format('Y/m/d H:i:s');
        }
    }

    /**
     * Get average order value for user from specific branch.
     */
    protected function getAverageOrderValue(int $userId, int $branchId, array $filters): ?float
    {
        $query = $this->model->where('user_id', $userId)
            ->where('branch_id', $branchId)
            ->where('payment_status', 'true')
            ->where('order_status', 'delivered');

        $this->applyFilters($query, $filters);

        return $query->avg('order_price');
    }

    /**
     * Get last order date for user from specific branch.
     */
    protected function getLastOrderDate(int $userId, int $branchId): ?string
    {
        $lastOrder = $this->model->where('user_id', $userId)
            ->where('branch_id', $branchId)
            ->where('order_status', 'delivered')
            ->orderBy('created_at', 'desc')
            ->first();

        return $lastOrder ? $lastOrder->created_at : null;
    }
}
