<?php

namespace App\Services;

use App\Models\DeliveryType;
use App\Helpers\PointInPolygonHelper;
use App\Repositories\Address\AddressRepository;
use Carbon\Carbon;

class AreaValidationService
{
    private const DELIVERY_METHOD_PRIORITY = [
        'restaurant' => 1,
        'platform' => 2,
        'default' => 3
    ];

    private AddressRepository $addressRepository;

    public function __construct(AddressRepository $addressRepository)
    {
        $this->addressRepository = $addressRepository;
    }

    /**
     * Validate area and get delivery information
     */
    public function validateAreaAndGetDeliveryInfo($result, $request): array
    {
        $deliveryType = $this->getPreferredDeliveryType($result->shipping_methods);
        $addressValid = $this->validateAddress($request, $result);

        return [
            'message' => 'رستوران در این زمان فعال میباشد',
            'data' => $result,
            'active' => true,
            'open_branch' => true,
            'delivery_type' => $deliveryType,
            'address_valid' => $addressValid
        ];
    }

    /**
     * Validate area and get delivery information with invalid address
     */
    public function validateAreaAndGetDeliveryInfoWithInvalidAddress($result, $request): array
    {
        $deliveryType = $this->getPreferredDeliveryType($result->shipping_methods);

        return [
            'message' => 'رستوران در این زمان فعال میباشد اما آدرس شما در محدوده ارسال قرار ندارد',
            'data' => $result,
            'active' => true,
            'open_branch' => true,
            'delivery_type' => $deliveryType,
            'address_valid' => false
        ];
    }

    /**
     * Get nearest available time response
     */
    public function getNearestAvailableTimeResponse($data): array
    {
        $deliveryType = $data ? $this->getPreferredDeliveryType($data->shipping_methods) : null;

        return [
            'message' => 'رستوران در این زمان فعال نمیباشد',
            'data' => $data,
            'active' => false,
            'open_branch' => (bool) $data,
            'delivery_type' => $deliveryType,
            'address_valid' => false
        ];
    }

    public function validateAreaWithNoAddress($result): array
    {
        return [
            'message' => 'رستوران فعال میباشد',
            'data' => null,
            'active' => true,
            'open_branch' => true,
            'delivery_type' => null,
            'address_valid' => false
        ];
    }

    /**
     * Handle case when no areas are registered for the branch
     */
    public function getNoAreasRegisteredResponse(): array
    {
        return [
            'message' => 'رستوران بسته است - هیچ منطقه‌ای ثبت نشده است',
            'data' => null,
            'active' => false,
            'open_branch' => false,
            'delivery_type' => null,
            'address_valid' => false
        ];
    }

    /**
     * Get holiday response with reason when restaurant is closed
     */
    public function getHolidayResponse($holiday): array
    {
        $reason = $holiday->reason ?? null;
        $reasonType = null;
        $reasonTypeLabel = null;

        if ($holiday->holiday_reason_type) {
            $reasonType = $holiday->holiday_reason_type->value;
            $reasonTypeLabel = $holiday->holiday_reason_type->getLabel();
        }

        $message = 'رستوران در این زمان تعطیل است';
        if ($reasonTypeLabel) {
            $message .= ' - ' . $reasonTypeLabel;
        } elseif ($reason) {
            $message .= ' - ' . $reason;
        }

        return [
            'message' => $message,
            'data' => null,
            'active' => false,
            'open_branch' => false,
            'delivery_type' => null,
            'address_valid' => false,
            'holiday' => [
                'reason' => $reason,
                'reason_type' => $reasonType,
                'reason_type_label' => $reasonTypeLabel,
            ]
        ];
    }

    /**
     * Get preferred delivery type based on priority
     */
    private function getPreferredDeliveryType($shippingMethods): ?DeliveryType
    {
        $shippingMethods = $this->normalizeShippingMethods($shippingMethods);
        $deliveryTypes = DeliveryType::whereIn('id', $shippingMethods)->get();

        return $deliveryTypes->sortBy(function ($item) {
            return self::DELIVERY_METHOD_PRIORITY[$item->delivery_method] ?? self::DELIVERY_METHOD_PRIORITY['default'];
        })->first();
    }

    /**
     * Normalize shipping methods to array
     */
    private function normalizeShippingMethods($shippingMethods): array
    {
        if (is_array($shippingMethods)) {
            return array_values(array_unique($shippingMethods));
        }

        $decoded = json_decode($shippingMethods, true);
        return is_array($decoded) ? array_values(array_unique($decoded)) : [];
    }

    /**
     * Validate address coordinates
     */
    private function validateAddress($request, $result): bool
    {
        if ($request->has('lat') && $request->has('lng')) {
            return $this->validateCoordinates($request->lat, $request->lng, $result->area);
        }

        if ($request->address_id) {
            return $this->validateAddressById($request->address_id, $result->area);
        }

        return false;
    }

    /**
     * Validate coordinates directly
     */
    private function validateCoordinates($lat, $lng, $area): bool
    {
        if (!$area || !isset($area->coordinates)) {
            return false;
        }

        $pointInPolygonHelper = app(PointInPolygonHelper::class);
        return $pointInPolygonHelper->check([$lat, $lng], $area->coordinates);
    }

    /**
     * Validate address by ID
     */
    private function validateAddressById($addressId, $area): bool
    {
        if (!$area || !isset($area->coordinates)) {
            return false;
        }

        $address = $this->addressRepository->find($addressId);
        if (!$address) {
            return false;
        }

        return PointInPolygonHelper::check([$address->lat, $address->lng], $area->coordinates);
    }
}
