<?php

namespace App\Services;

use App\Models\CustomerBranch;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CustomerService
{
    /**
     * Get all customers for a specific branch
     */
    public function getCustomersForBranch($branchId, $search = '', $perPage = 20)
    {
        try {
            $query = CustomerBranch::with(['user'])
                ->where('branch_id', $branchId);

            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('family', 'like', "%{$search}%")
                      ->orWhere('mobile', 'like', "%{$search}%");
                });
            }

            return $query->orderBy('created_at', 'desc')->paginate($perPage);
        } catch (\Exception $e) {
            Log::error('Error in CustomerService::getCustomersForBranch: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get a specific customer by ID
     */
    public function getCustomerById($id)
    {
        try {
            return CustomerBranch::with(['user', 'branch'])->findOrFail($id);
        } catch (\Exception $e) {
            Log::error('Error in CustomerService::getCustomerById: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Find or create user by mobile number
     */
    private function findOrCreateUser($data)
    {
        $existingUser = User::where('mobile', $data['mobile'])->first();

        if ($existingUser) {
            // Update user information if there are changes
            $userData = [];
            if (isset($data['name']) && $data['name'] !== $existingUser->name) {
                $userData['name'] = $data['name'];
            }
            if (isset($data['family']) && $data['family'] !== $existingUser->family) {
                $userData['family'] = $data['family'];
            }
            if (isset($data['email']) && $data['email'] !== $existingUser->email) {
                $userData['email'] = $data['email'];
            }

            if (!empty($userData)) {
                $existingUser->update($userData);
            }

            return $existingUser;
        }

        // Create new user if doesn't exist
        return User::create([
            'name' => $data['name'],
            'family' => $data['family'],
            'mobile' => $data['mobile'],
            'email' => $data['email'] ?? null,
            'role_id' => 4, // Client role
            'target_role_id' => 0
        ]);
    }

    /**
     * Create a new customer
     */
    public function createCustomer($data)
    {
        try {
            DB::beginTransaction();

            // Check if customer already exists for this branch
            $existingCustomer = CustomerBranch::where('mobile', $data['mobile'])
                ->where('branch_id', $data['branch_id'])
                ->first();

            if ($existingCustomer) {
                throw new \Exception('Customer already exists for this branch');
            }

            // Find or create user
            $user = $this->findOrCreateUser($data);

            // Create customer branch relationship
            $customerBranch = CustomerBranch::create([
                'name' => $data['name'],
                'family' => $data['family'],
                'user_id' => $user->id,
                'branch_id' => $data['branch_id'],
                'mobile' => $data['mobile'],
                'email' => $data['email'] ?? null,
                'address' => $data['address'] ?? null,
                'status' => 'active'
            ]);

            DB::commit();

            return $customerBranch->load('user');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error in CustomerService::createCustomer: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update an existing customer
     */
    public function updateCustomer($id, $data)
    {
        try {
            DB::beginTransaction();

            $customer = CustomerBranch::findOrFail($id);

            // Update user if needed
            if (isset($data['name']) || isset($data['family']) || isset($data['mobile']) || isset($data['email'])) {
                $userData = [];
                if (isset($data['name'])) $userData['name'] = $data['name'];
                if (isset($data['family'])) $userData['family'] = $data['family'];
                if (isset($data['mobile'])) $userData['mobile'] = $data['mobile'];
                if (isset($data['email'])) $userData['email'] = $data['email'];

                $customer->user->update($userData);
            }

            // Update customer branch
            $customer->update($data);

            DB::commit();

            return $customer->load('user');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error in CustomerService::updateCustomer: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Delete a customer
     */
    public function deleteCustomer($id)
    {
        Log::info($id);
        Log::info('12312312321');
        try {
            DB::beginTransaction();

            // First check if customer exists
            $customer = CustomerBranch::find($id);

            if (!$customer) {
                throw new \Exception("Customer with ID {$id} not found");
            }

            Log::info("Deleting customer: ID={$id}, Name={$customer->name}, Branch={$customer->branch_id}");

            // Soft delete the customer branch relationship
            $customer->delete();

            Log::info("Customer branch relationship deleted successfully");

            // Also soft delete the user if they don't have other relationships
            $otherRelationships = CustomerBranch::where('user_id', $customer->user_id)->count();

            if ($otherRelationships == 0) {
                $customer->user->delete();
                Log::info("User also deleted as no other relationships exist");
            } else {
                Log::info("User kept as {$otherRelationships} other relationships exist");
            }

            DB::commit();
            Log::info("Customer deletion completed successfully");

            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error in CustomerService::deleteCustomer: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    /**
     * Search customers by name, family, or mobile
     */
    public function searchCustomers($branchId, $search)
    {
        try {
            $query = CustomerBranch::with(['user'])
                ->where('branch_id', $branchId)
                ->active();

            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('family', 'like', "%{$search}%")
                      ->orWhere('mobile', 'like', "%{$search}%");
                });
            }

            return $query->orderBy('created_at', 'desc')->paginate(20);
        } catch (\Exception $e) {
            Log::error('Error in CustomerService::searchCustomers: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get customer statistics for a branch
     */
    public function getCustomerStats($branchId)
    {
        try {
            $totalCustomers = CustomerBranch::where('branch_id', $branchId)->count();
            $activeCustomers = CustomerBranch::where('branch_id', $branchId)->active()->count();
            $inactiveCustomers = $totalCustomers - $activeCustomers;

            return [
                'total' => $totalCustomers,
                'active' => $activeCustomers,
                'inactive' => $inactiveCustomers
            ];
        } catch (\Exception $e) {
            Log::error('Error in CustomerService::getCustomerStats: ' . $e->getMessage());
            throw $e;
        }
    }
}
