<?php

namespace App\Services;


use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Enums\PaymentType;
use App\Enums\TransactionType;
use App\Events\OrderChangeStatus;
use App\Models\UserBranchWallet;
use App\Repositories\Order\OrderRepository;
use App\Repositories\Payment\PaymentGateways\BitPayGateway;
use App\Repositories\Payment\PaymentGateways\MeliBankGateway;
use App\Repositories\Payment\PaymentGateways\MellatGateway;
use App\Repositories\Payment\PaymentGateways\OmidPayGateway;
use App\Repositories\Payment\PaymentGateways\SepGateway;
use App\Repositories\Payment\PaymentGateways\WalletGateway;
use App\Repositories\Payment\PaymentGateways\ZarinPalGateway;
use App\Repositories\Payment\PaymentGateways\ZibalGateway;
use App\Repositories\Payment\PaymentRepository;
use App\Repositories\Transaction\TransactionRepository;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\BankAccount;
use App\Models\User;
use App\Repositories\Branch\BranchRepository;
use App\Repositories\Setting\SettingRepository;


class PaymentService
{
    protected $paymentRepository;
    protected $orderRepository;
    protected $gateway;

    public function __construct(OrderRepository $orderRepository)
    {
//        $this->paymentRepository = $paymentRepository;
        $this->orderRepository = $orderRepository;
    }

    public function setGateway(string $gatewayId)
    {

        $gateway = app(PaymentRepository::class)->getGatewayById($gatewayId);
        $gateway = $gateway->name ?? null;
        switch ($gateway) {
            case 'zarinpal':
                $this->gateway = new ZarinPalGateway();
                break;
            case 'bitPay':
                $this->gateway = new BitPayGateway();
                break;
            case 'MeliBankGateway':
                $this->gateway = new MeliBankGateway();
                break;
            case 'OMIDPAY':
                $this->gateway = new OmidPayGateway();
                break;
            case 'SEPPAY':
                $this->gateway = new SepGateway();
                break;
            case 'ZIBAl':
                $this->gateway = new ZibalGateway();
                break;
            case 'MELLAT':
                $this->gateway = new MellatGateway();
                break;
            default:
                $this->gateway = new ZibalGateway();
        }
    }

    public function processPayment($paymentCode, $useWallet, $gatewayId)
    {
        DB::beginTransaction();
        try {
            $order = $this->orderRepository->getByPaymentCode($paymentCode);


            if ($order == null)
                return ['status' => 200, 'message' => 'عملیات با شکست روبرو شد ', 'data' => ['route' => env('FrontUrl') . 'order?status=400&orderid=' . 0]];


            if ($order->payment_status == "true") {
                return ['status' => 200, 'data' => ['route' => env('FrontUrl') . 'order?status=200&orderid=' . $order->id], 'message' => 'با موفقیت انجام شد '];
            }


            $this->setGateway($gatewayId);


            if (!$this->gateway) {
                return ['status' => 200, 'message' => 'عملیات با شکست روبرو شد ', 'data' => ['route' => env('FrontUrl') . 'order?status=400&orderid=' . $order->id]];
            }

            $totalPrice = $order->order_price;
            $totalPriceWithoutShipping = $order->order_price - $order->delivery_price;
            $cashBack = 0;
            $userBranchWallet = UserBranchWallet::where('user_id', $order->user_id)->where('branch_id', $order->branch_id)->first();
            if ($userBranchWallet != null and $userBranchWallet->amount > 0) {
                $cashBack = $userBranchWallet->amount;
                if ($cashBack > $totalPriceWithoutShipping) {
                    $cashBack = $totalPriceWithoutShipping;
                }

                $totalPrice = $totalPrice - $cashBack;
            }

            if ($useWallet) {
                $walletGateway = app(WalletGateway::class)->setPay($order->payment_code, $totalPrice);
                if ($walletGateway['data']['walletAmount'] == $totalPrice) {
                    if ($cashBack > 0) {
                        $userBranchWallet = UserBranchWallet::where('user_id', $order->user_id)->where('branch_id', $order->branch_id)->first();
                        $userBranchWallet->update(['amount' => $userBranchWallet->amount - $cashBack]);
                    }

                    $order->update(['wallet_use_price' => $walletGateway['data']['walletAmount'], 'cash_back_price' => $cashBack, 'payment_type' => PaymentType::WALLET, "payment_status" => PaymentStatus::TRUE, 'order_status' => OrderStatus::AWAITINGRESTAURANTAPPRAVAL]);
                    event(new OrderChangeStatus($order));
                    return $walletGateway;
                } elseif ($walletGateway['data']['walletAmount'] > 0) {
                    $totalPrice = $totalPrice - $walletGateway['data']['walletAmount'];
                }
            }

            $multiplexingInfos = [];
            if ($this->gateway instanceof ZibalGateway) {
                $branch = app(BranchRepository::class)->show($order->branch_id);
                $vendorUser = User::where('target_role_id', $order->branch_id)->first();
                $bankAccount = BankAccount::where('user_id', $vendorUser->id)->where('is_active', true)->first();

                if ($bankAccount && !empty($bankAccount->shaba_number)) {
                    if ($branch->percent_order == null || (int)$branch->percent_order==0) {
                        $percentageSales = app(SettingRepository::class)->getSettingWithName('percentageSales')['value'];
                    } else {
                        $percentageSales = $branch->percent_order;
                    }
                    $orderPriceM = $order->order_price - $order->delivery_price;
                    $sellerShare = PriceCalculation($orderPriceM, $percentageSales);
                    // Zibal expects amount in Rial (multiplexingInfos amounts sum up to total amount * 10)
                    // But here we are passing Toman to setPay, and setPay converts to Rial (* 10).
                    // So we should pass Toman here if we modify setPay to handle it, OR we pass Rial and ZibalGateway handles it.
                    // Let's look at ZibalGateway::setPay: "amount" => $amount * 10
                    // So we should pass Toman here.
                    
                    $shaba = $bankAccount->shaba_number;
                    // if (!str_starts_with(strtoupper($shaba), 'IR')) {
                    //     $shaba = 'IR' . $shaba;
                    // }

                    $multiplexingInfos = [
                        [
                            "bankAccount" => $shaba,
                            "amount" => (int)$sellerShare * 10 // Zibal expects Rial in multiplexingInfos
                        ],
                        [
                            "bankAccount" => "IR400600301270011192650001",
                            "amount" => ((int)$order->order_price*10 - (int)$sellerShare * 10) / 2 // Zibal expects Rial in multiplexingInfos
                        ]                        ,
                        [
                            "bankAccount" => "IR480600301570011090140001",
                            "amount" => ((int)$order->order_price*10 - (int)$sellerShare * 10)/2 // Zibal expects Rial in multiplexingInfos
                        ]
                    ];
                }
            }

            $order->update(['wallet_use_price' => $walletGateway['data']['walletAmount'] ?? 0, 'cash_back_price' => $cashBack]);
            $set = $this->gateway->setPay($order->payment_code, $totalPrice, null, $order->id, ['multiplexingInfos' => $multiplexingInfos]);
            if ($set['status'] == 200) {
                $order->update(['payment_code_gateway' => (string)$set['data']['codeGateway']]);
                return $set;
            }
            return ['status' => 500, 'message' => 'عملیات با شکست روبرو شد ', 'data' => null];
        } catch (\Exception $e) {
            handelError('error', $e->getMessage());
            DB::rollBack();
            return ['status' => 500, 'message' => 'عملیات با شکست روبرو شد ', 'data' => null];

        }
    }

    public function confirmPayment(array $data): array
    {
        DB::beginTransaction();
        try {
            $this->setGateway($data['gateway_id']);
            $checkConfirm = $this->gateway->confirmPay($data);
            $order = $checkConfirm['data']['order'] ?? [];

            if ($checkConfirm['status'] == 200) {
                //sub cash

                if ($order->cash_back_price > 0) {
                    $userBranchWallet = UserBranchWallet::where('user_id', $order->user_id)->where('branch_id', $order->branch_id)->first();
                    $userBranchWallet->update(['amount' => $userBranchWallet->amount - $order->cash_back_price]);
                }
                $order->update([
                    "payment_status" => PaymentStatus::TRUE,
                    "order_status" => OrderStatus::AWAITINGRESTAURANTAPPRAVAL,
                    "payment_code_gateway" => $checkConfirm['data']['refId']
                ]);
                event(new OrderChangeStatus($order));
                DB::commit();
                return ['status' => 200, 'message' => '', 'data' => ['orderId' => $order->id, "paymentCodeGateway" => $checkConfirm['data']['refId']]];
            }

            if ($order->wallet_use_price != 0)
                app(TransactionRepository::class)->storeP(
                    [
                        'amount' => $order->wallet_use_price,
                        'user_id' => $order->user_id,
                        'transaction_type' => TransactionType::PAYORDER,
                        'target_id' => $order->branch_id,
                        'description' => 'بازگشت پول بابت شکست در عملیات پرداخت'
                    ]);

            DB::commit();
            return ['status' => 500, 'message' => 'عملیات باشکست روبرو شد.', 'data' => ['orderId' => $order->id ?? 0]];
        } catch (\Exception $e) {
            dd($e);
            handelError('error', $e);
            DB::rollBack();
            return ['status' => 500, 'message' => 'عملیات باشکست روبرو شد.', 'data' => ['orderId' => 0]];

        }
    }

    /**
     * ساخت لینک پرداخت برای سفارش
     *
     * @param int $userId
     * @param int $amount
     * @param array $meta
     * @return string
     */
    public function createPaymentLink($userId, $amount, $meta = [])
    {
        // اینجا می‌توانی منطق اتصال به درگاه واقعی را بنویسی
        // فعلاً یک لینک فرضی برمی‌گردانیم
        $orderId = $meta['order_id'] ?? 0;
        return url("/pay?user_id={$userId}&order_id={$orderId}&amount={$amount}");
    }
}
