<?php

namespace App\Services;

use App\Repositories\UserOrder\UserOrderRepositoryInterface;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Log;

class UserOrderService
{
    protected UserOrderRepositoryInterface $repository;

    public function __construct(UserOrderRepositoryInterface $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Get user's order history from a specific branch.
     *
     * @param int $userId
     * @param int $branchId
     * @param array $filters
     * @param int $perPage
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getUserBranchOrders(int $userId, int $branchId, array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        try {
            // Validate user exists
            $this->validateUser($userId);

            // Validate branch exists
            $this->validateBranch($branchId);

            return $this->repository->getUserBranchOrders($userId, $branchId, $filters, $perPage);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت تاریخچه خرید کاربر: ' . $e->getMessage(), [
                'user_id' => $userId,
                'branch_id' => $branchId,
                'filters' => $filters,
                'per_page' => $perPage
            ]);
            throw $e;
        }
    }

    /**
     * Get user's order statistics from a specific branch.
     *
     * @param int $userId
     * @param int $branchId
     * @param array $filters
     * @return array
     * @throws \Exception
     */
    public function getUserBranchOrderStats(int $userId, int $branchId, array $filters = []): array
    {
        try {
            // Validate user exists
            $this->validateUser($userId);

            // Validate branch exists
            $this->validateBranch($branchId);

            return $this->repository->getUserBranchOrderStats($userId, $branchId, $filters);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت آمار خرید کاربر: ' . $e->getMessage(), [
                'user_id' => $userId,
                'branch_id' => $branchId,
                'filters' => $filters
            ]);
            throw $e;
        }
    }

    /**
     * Get user's favorite products from a specific branch.
     *
     * @param int $userId
     * @param int $branchId
     * @param int $limit
     * @return array
     * @throws \Exception
     */
    public function getUserFavoriteProducts(int $userId, int $branchId, int $limit = 10): array
    {
        try {
            // Validate user exists
            $this->validateUser($userId);

            // Validate branch exists
            $this->validateBranch($branchId);

            return $this->repository->getUserFavoriteProducts($userId, $branchId, $limit);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت محصولات محبوب کاربر: ' . $e->getMessage(), [
                'user_id' => $userId,
                'branch_id' => $branchId,
                'limit' => $limit
            ]);
            throw $e;
        }
    }

    /**
     * Validate user exists and is active.
     *
     * @param int $userId
     * @return bool
     * @throws \Exception
     */
    protected function validateUser(int $userId): bool
    {
        $user = \App\Models\User::where('id', $userId)
            ->whereNull('deleted_at')
            ->first();

        if (!$user) {
            throw new \Exception('کاربر یافت نشد یا غیرفعال است');
        }

        return true;
    }

    /**
     * Validate branch exists and is active.
     *
     * @param int $branchId
     * @return bool
     * @throws \Exception
     */
    protected function validateBranch(int $branchId): bool
    {
        $branch = \App\Models\Branch::where('id', $branchId)
            ->where('status', 'active')
            ->whereNull('deleted_at')
            ->first();

        if (!$branch) {
            throw new \Exception('فروشگاه یافت نشد یا غیرفعال است');
        }

        return true;
    }

    /**
     * Get user and branch information.
     *
     * @param int $userId
     * @param int $branchId
     * @return array
     * @throws \Exception
     */
    public function getUserBranchInfo(int $userId, int $branchId): array
    {
        try {
            $user = \App\Models\User::select('id', 'name', 'family', 'mobile', 'email')
                ->where('id', $userId)
                ->whereNull('deleted_at')
                ->first();

            $branch = \App\Models\Branch::select('id', 'title', 'address', 'logo', 'min_order', 'rate')
                ->where('id', $branchId)
                ->where('status', 'active')
                ->whereNull('deleted_at')
                ->first();

            if (!$user) {
                throw new \Exception('کاربر یافت نشد');
            }

            if (!$branch) {
                throw new \Exception('فروشگاه یافت نشد');
            }

            return [
                'user' => $user,
                'branch' => $branch
            ];
        } catch (\Exception $e) {
            Log::error('خطا در دریافت اطلاعات کاربر و فروشگاه: ' . $e->getMessage(), [
                'user_id' => $userId,
                'branch_id' => $branchId
            ]);
            throw $e;
        }
    }
}
