<?php

namespace App\Services;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Enums\PaymentType;
use App\Enums\TransactionType;
use App\Events\OrderChangeStatus;
use App\Models\ChargeWallet;
use App\Models\UserBranchWallet;
use App\Repositories\Order\OrderRepository;
use App\Repositories\Payment\PaymentGateways\BitPayGateway;
use App\Repositories\Payment\PaymentGateways\MeliBankGateway;
use App\Repositories\Payment\PaymentGateways\MellatGateway;
use App\Repositories\Payment\PaymentGateways\WalletGateway;
use App\Repositories\Payment\PaymentGateways\ZarinPalGateway;
use App\Repositories\Payment\PaymentRepository;
use App\Repositories\Transaction\TransactionRepository;
use App\Repositories\User\UserRepository;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class WalletService
{
    protected $userRepository;
    protected $gateway;

    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    public function setGateway(string $gatewayId)
    {
        $gateway = app(PaymentRepository::class)->getGatewayById($gatewayId);
        $gateway = $gateway->name ?? null;
        switch ($gateway) {
            case 'zarinpal':
                $this->gateway = new ZarinPalGateway();
                break;
            case 'bitPay':
                $this->gateway = new BitPayGateway();
                break;
            case 'MeliBankGateway':
                $this->gateway = new MeliBankGateway();
                break;
            case 'mellat':
                $this->gateway = new MellatGateway();
                break;
            default:
                $this->gateway = new ZarinPalGateway();
        }
    }

    public function processPayment($paymentCode, $gatewayId)
    {
        DB::beginTransaction();
        try {
            $this->setGateway($gatewayId);
            $callbackRoute = route('chargeWalletConfirm', ['gateway_id' => $gatewayId]);
            $chargeWallet = ChargeWallet::where('code_pay', $paymentCode)->first();
            if ($chargeWallet != null) {
                $amount = $chargeWallet->amount;
                $result = $this->gateway->setPay($paymentCode, $amount, $callbackRoute);
                if ($result['status'] == 200) {
                    ChargeWallet::where('id', $chargeWallet->id)->update(['payment_code_gateway' => $result['data']['codeGateway']]);
                    DB::commit();
                    return $result;
                }
            }
            DB::rollBack();
            return ['status' => 500, 'message' => 'عملیات با شکست روبرو شد ', 'data' => null];
        } catch (\Exception $e) {
            handelError('error', $e->getMessage());
            DB::rollBack();
            return ['status' => 500, 'message' => 'عملیات با شکست روبرو شد ', 'data' => null];
        }
    }

    public function confirmPayment(array $data)
    {
        DB::beginTransaction();
        try {
            $this->setGateway($data['gateway_id']);
            $checkConfirm = $this->gateway->confirmChargePay($data);
            $order = $checkConfirm['data']['order'] ?? [];
            if ($checkConfirm['status'] == 200) {
                // sub cash
                ChargeWallet::where('payment_code_gateway', $order->payment_code_gateway)->update([
                    'status' => "true"
                ]);
                $result = app(TransactionRepository::class)->storeP([
                    'amount' => $checkConfirm['data']['order']->amount ?? 0,
                    'user_id' => $checkConfirm['data']['order']->user_id ?? 0,
                    'transaction_type' => TransactionType::CHARGE
                ]);
                if ($result == null) {
                    DB::rollBack();
                    throw new \Exception('error transaction');
                }
                DB::commit();
                return ['status' => 200, 'message' => '', 'data' => ['orderId' => $order->id, "paymentCodeGateway" => $checkConfirm['data']['refId']]];
            }
            DB::rollBack();
            return ['status' => 500, 'message' => 'عملیات باشکست روبرو شد.', 'data' => ['orderId' => 0]];
        } catch (\Exception $e) {
            DB::rollBack();
            handelError('error', $e->getMessage());
            return ['status' => 500, 'message' => 'عملیات باشکست روبرو شد. خطا: ' . $e->getMessage(), 'data' => ['orderId' => 0]];
        }
    }

    public function increase($userId, $amount, $description)
    {
        $chargeWallets = ChargeWallet::create([
            'user_id' => $userId,
            'amount' => $amount,
            'description' => $description,
            'code_pay' => Str::random(10),
            'status' => 'true',
            'payment_code_gateway' => 'wallet',
        ]);
        return $chargeWallets;
    }
}
