<?php

namespace App\Services;

use App\Enums\CouponType;
use App\Models\Branch;
use App\Models\Coupon;
use App\Models\Setting;
use App\Repositories\Coupon\CouponRepository;
use App\Repositories\SMS\SMSRepositoryInterface;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class WelcomeCouponService
{
    protected $couponRepository;
    protected $smsRepository;

    public function __construct(CouponRepository $couponRepository, SMSRepositoryInterface $smsRepository)
    {
        $this->couponRepository = $couponRepository;
        $this->smsRepository = $smsRepository;
    }

    /**
     * Create welcome coupon for new user
     *
     * @param int $userId
     * @param string $mobile
     * @return bool
     */
    public function createWelcomeCouponForUser($userId, $mobile)
    {
        try {
            // Check if user already has a welcome coupon
            $existingCoupon = DB::table('coupon_users')
                ->join('coupons', 'coupon_users.coupon_id', '=', 'coupons.id')
                ->where('coupon_users.user_id', $userId)
                ->where('coupons.title', 'like', '%خوش%آمدگویی%')
                ->whereNull('coupons.deleted_at')
                ->first();

            if ($existingCoupon) {
                Log::info("User {$userId} already has a welcome coupon");
                return false;
            }

            // Get welcome coupon settings
            $settings = $this->getWelcomeCouponSettings();
            
            if (!$settings || !$this->isWelcomeCouponEnabled($settings)) {
                Log::info("Welcome coupon is not enabled or settings are incomplete");
                return false;
            }

            // Generate unique coupon code
            $couponCode = $this->generateUniqueCouponCode();

            // Calculate validity period
            $validityHours = (int)($settings['welcome_coupon_validity_hours'] ?? 24);
            $activeFrom = Carbon::now();
            $activeTo = Carbon::now()->addHours($validityHours);

            // Get all active branches
            $allBranches = Branch::where('status', '!=', 'disable')->pluck('id')->toArray();

            // Prepare coupon data
            $couponType = (int)($settings['welcome_coupon_type'] ?? 1); // 0 = Fixed, 1 = Percentage
            $couponValue = (float)($settings['welcome_coupon_value'] ?? 0);
            
            $couponData = [
                'title' => 'کوپن خوش‌آمدگویی',
                'code' => $couponCode,
                'branch_id' => 0, // 0 means all branches
                'type' => $couponType,
                'percent' => $couponType == 1 ? $couponValue : 0,
                'price' => $couponType == 0 ? $couponValue : 0,
                'active_from' => $activeFrom,
                'active_to' => $activeTo,
                'limit_uses' => !empty($allBranches) ? count($allBranches) : 1, // Can be used once per branch
                'limit_user' => 'true',
                'limit_branch' => !empty($allBranches) ? 'true' : 'false',
                'used_count' => 0,
                'max_price_discount' => $settings['welcome_coupon_max_discount'] ?? null,
                'min_price_order' => $settings['welcome_coupon_min_order'] ?? null,
                'condition_type' => CouponType::PUBLIC->value,
                'percent_system' => 0,
                'percent_branch' => 0,
            ];

            // Add user_ids and branch_ids to coupon data for storeA method
            $couponData['user_ids'] = [$userId];
            $couponData['branch_ids'] = !empty($allBranches) ? $allBranches : [];

            // Create coupon
            $coupon = $this->couponRepository->storeA($couponData);

            if (!$coupon) {
                Log::error("Failed to create welcome coupon for user {$userId}");
                return false;
            }

            // Send SMS with coupon code
            $this->smsRepository->sendSMSWelcomeCoupon($couponCode, $mobile);

            Log::info("Welcome coupon created successfully for user {$userId} with code {$couponCode}");
            return true;

        } catch (\Exception $e) {
            Log::error("Error creating welcome coupon: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get welcome coupon settings
     *
     * @return array|null
     */
    protected function getWelcomeCouponSettings()
    {
        $settings = Setting::whereIn('name', [
            'welcome_coupon_type',
            'welcome_coupon_value',
            'welcome_coupon_max_discount',
            'welcome_coupon_min_order',
            'welcome_coupon_validity_hours'
        ])->whereNull('branch_id')
            ->pluck('value', 'name')
            ->toArray();

        return $settings;
    }

    /**
     * Check if welcome coupon is enabled
     *
     * @param array $settings
     * @return bool
     */
    protected function isWelcomeCouponEnabled($settings)
    {
        return isset($settings['welcome_coupon_type']) &&
               isset($settings['welcome_coupon_value']) &&
               !empty($settings['welcome_coupon_value']) &&
               (float)$settings['welcome_coupon_value'] > 0;
    }

    /**
     * Generate unique coupon code
     *
     * @return string
     */
    protected function generateUniqueCouponCode()
    {
        do {
            $code = strtoupper(Str::random(8));
        } while (Coupon::where('code', $code)->exists());

        return $code;
    }
}

