# API Documentation - Area Special Conditions

## Overview
This document describes the API endpoints for managing area special conditions in the LimooFood system. Special conditions allow setting specific discounts or special offers for certain areas during specific time periods.

## Base URL
All endpoints are prefixed with `/api/v1/admin/area-special-conditions`

## Authentication
All endpoints require authentication using Bearer token in the Authorization header:
```
Authorization: Bearer {your_token}
```

## Data Model

### AreaSpecialCondition
```json
{
  "id": 1,
  "branch_id": 5,
  "area_id": 10,
  "title": "تخفیف ویژه آخر هفته",
  "description": "تخفیف 20 درصدی برای سفارشات آخر هفته",
  "start_time": "18:00",
  "end_time": "23:59",
  "date": "2024-01-15",
  "percent": 20,
  "status": "active",
  "created_at": "2024-01-10T10:00:00.000000Z",
  "updated_at": "2024-01-10T10:00:00.000000Z",
  "branch": {
    "id": 5,
    "title": "رستوران مرکزی"
  },
  "area": {
    "id": 10,
    "title": "منطقه 1"
  }
}
```

## API Endpoints

### 1. Get All Special Conditions
**GET** `/api/v1/admin/area-special-conditions/`

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `branch_id` | integer | No | Filter by branch ID |
| `area_id` | integer | No | Filter by area ID |
| `date` | date | No | Filter by specific date (YYYY-MM-DD) |
| `start_date` | date | No | Filter from date (YYYY-MM-DD) |
| `end_date` | date | No | Filter to date (YYYY-MM-DD) |
| `status` | string | No | Filter by status (active/inactive) |
| `search` | string | No | Search in title, description, branch title, or area title |
| `per_page` | integer | No | Number of items per page (1-100, default: 15) |
| `sort_column` | string | No | Column to sort by (default: created_at) |
| `sort_type` | string | No | Sort direction (asc/desc, default: desc) |

#### Example Request
```bash
GET /api/v1/admin/area-special-conditions/?branch_id=5&status=active&per_page=10
```

#### Example Response
```json
{
  "message": "لیست شرایط ویژه با موفقیت دریافت شد",
  "data": {
    "current_page": 1,
    "data": [
      {
        "id": 1,
        "branch_id": 5,
        "area_id": 10,
        "title": "تخفیف ویژه آخر هفته",
        "description": "تخفیف 20 درصدی برای سفارشات آخر هفته",
        "start_time": "18:00",
        "end_time": "23:59",
        "date": "2024-01-15",
        "percent": 20,
        "status": "active",
        "created_at": "2024-01-10T10:00:00.000000Z",
        "updated_at": "2024-01-10T10:00:00.000000Z",
        "branch": {
          "id": 5,
          "title": "رستوران مرکزی"
        },
        "area": {
          "id": 10,
          "title": "منطقه 1"
        }
      }
    ],
    "first_page_url": "http://localhost/api/v1/admin/area-special-conditions?page=1",
    "from": 1,
    "last_page": 1,
    "last_page_url": "http://localhost/api/v1/admin/area-special-conditions?page=1",
    "links": [...],
    "next_page_url": null,
    "path": "http://localhost/api/v1/admin/area-special-conditions",
    "per_page": 10,
    "prev_page_url": null,
    "to": 1,
    "total": 1
  },
  "status": true
}
```

### 2. Create Special Condition
**POST** `/api/v1/admin/area-special-conditions/store`

#### Request Body
```json
{
  "branch_id": 5,
  "area_id": 10,
  "title": "تخفیف ویژه آخر هفته",
  "description": "تخفیف 20 درصدی برای سفارشات آخر هفته",
  "start_time": "18:00",
  "end_time": "23:59",
  "date": "2024-01-15",
  "percent": 20,
  "status": "active"
}
```

#### Validation Rules
| Field | Type | Required | Validation |
|-------|------|----------|------------|
| `branch_id` | integer | Yes | Must exist in branches table |
| `area_id` | integer | Yes | Must exist in areas table |
| `title` | string | No | Max 255 characters |
| `description` | string | No | Max 1000 characters |
| `start_time` | string | Yes | Format: HH:MM |
| `end_time` | string | Yes | Format: HH:MM, must be after start_time |
| `date` | date | Yes | Must be today or future date |
| `percent` | integer | Yes | Between 0-100 |
| `status` | string | No | active or inactive (default: active) |

#### Example Response
```json
{
  "message": "شرایط ویژه با موفقیت ایجاد شد",
  "data": {
    "id": 1,
    "branch_id": 5,
    "area_id": 10,
    "title": "تخفیف ویژه آخر هفته",
    "description": "تخفیف 20 درصدی برای سفارشات آخر هفته",
    "start_time": "18:00",
    "end_time": "23:59",
    "date": "2024-01-15",
    "percent": 20,
    "status": "active",
    "created_at": "2024-01-10T10:00:00.000000Z",
    "updated_at": "2024-01-10T10:00:00.000000Z",
    "branch": {
      "id": 5,
      "title": "رستوران مرکزی"
    },
    "area": {
      "id": 10,
      "title": "منطقه 1"
    }
  },
  "status": true
}
```

### 3. Get Specific Special Condition
**GET** `/api/v1/admin/area-special-conditions/show`

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `id` | integer | Yes | Special condition ID |

#### Example Request
```bash
GET /api/v1/admin/area-special-conditions/show?id=1
```

#### Example Response
```json
{
  "message": "شرایط ویژه با موفقیت دریافت شد",
  "data": {
    "id": 1,
    "branch_id": 5,
    "area_id": 10,
    "title": "تخفیف ویژه آخر هفته",
    "description": "تخفیف 20 درصدی برای سفارشات آخر هفته",
    "start_time": "18:00",
    "end_time": "23:59",
    "date": "2024-01-15",
    "percent": 20,
    "status": "active",
    "created_at": "2024-01-10T10:00:00.000000Z",
    "updated_at": "2024-01-10T10:00:00.000000Z",
    "branch": {
      "id": 5,
      "title": "رستوران مرکزی"
    },
    "area": {
      "id": 10,
      "title": "منطقه 1"
    }
  },
  "status": true
}
```

### 4. Update Special Condition
**PUT** `/api/v1/admin/area-special-conditions/update`

#### Request Body
```json
{
  "id": 1,
  "branch_id": 5,
  "area_id": 10,
  "title": "تخفیف ویژه آخر هفته - به‌روزرسانی شده",
  "description": "تخفیف 25 درصدی برای سفارشات آخر هفته",
  "start_time": "17:00",
  "end_time": "23:59",
  "date": "2024-01-15",
  "percent": 25,
  "status": "active"
}
```

#### Example Response
```json
{
  "message": "شرایط ویژه با موفقیت به‌روزرسانی شد",
  "data": {
    "id": 1,
    "branch_id": 5,
    "area_id": 10,
    "title": "تخفیف ویژه آخر هفته - به‌روزرسانی شده",
    "description": "تخفیف 25 درصدی برای سفارشات آخر هفته",
    "start_time": "17:00",
    "end_time": "23:59",
    "date": "2024-01-15",
    "percent": 25,
    "status": "active",
    "created_at": "2024-01-10T10:00:00.000000Z",
    "updated_at": "2024-01-10T11:30:00.000000Z",
    "branch": {
      "id": 5,
      "title": "رستوران مرکزی"
    },
    "area": {
      "id": 10,
      "title": "منطقه 1"
    }
  },
  "status": true
}
```

### 5. Delete Special Condition
**DELETE** `/api/v1/admin/area-special-conditions/destroy`

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `id` | integer | Yes | Special condition ID |

#### Example Request
```bash
DELETE /api/v1/admin/area-special-conditions/destroy?id=1
```

#### Example Response
```json
{
  "message": "شرایط ویژه با موفقیت حذف شد",
  "data": [],
  "status": true
}
```

### 6. Get Special Conditions by Branch
**GET** `/api/v1/admin/area-special-conditions/by-branch`

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `branch_id` | integer | Yes | Branch ID |
| `date` | date | No | Filter by specific date |
| `status` | string | No | Filter by status (active/inactive) |

#### Example Request
```bash
GET /api/v1/admin/area-special-conditions/by-branch?branch_id=5&status=active
```

### 7. Get Special Conditions by Area
**GET** `/api/v1/admin/area-special-conditions/by-area`

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `area_id` | integer | Yes | Area ID |
| `date` | date | No | Filter by specific date |
| `status` | string | No | Filter by status (active/inactive) |

#### Example Request
```bash
GET /api/v1/admin/area-special-conditions/by-area?area_id=10&status=active
```

### 8. Get Currently Active Special Conditions
**GET** `/api/v1/admin/area-special-conditions/currently-active`

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `branch_id` | integer | No | Filter by branch ID |
| `area_id` | integer | No | Filter by area ID |

#### Example Request
```bash
GET /api/v1/admin/area-special-conditions/currently-active?branch_id=5
```

#### Example Response
```json
{
  "message": "شرایط ویژه فعال فعلی با موفقیت دریافت شد",
  "data": [
    {
      "id": 1,
      "branch_id": 5,
      "area_id": 10,
      "title": "تخفیف ویژه آخر هفته",
      "description": "تخفیف 20 درصدی برای سفارشات آخر هفته",
      "start_time": "18:00",
      "end_time": "23:59",
      "date": "2024-01-15",
      "percent": 20,
      "status": "active",
      "created_at": "2024-01-10T10:00:00.000000Z",
      "updated_at": "2024-01-10T10:00:00.000000Z",
      "branch": {
        "id": 5,
        "title": "رستوران مرکزی"
      },
      "area": {
        "id": 10,
        "title": "منطقه 1"
      }
    }
  ],
  "status": true
}
```

### 9. Toggle Special Condition Status
**POST** `/api/v1/admin/area-special-conditions/toggle-status`

#### Request Body
```json
{
  "id": 1
}
```

#### Example Response
```json
{
  "message": "وضعیت شرایط ویژه به inactive تغییر یافت",
  "data": {
    "id": 1,
    "branch_id": 5,
    "area_id": 10,
    "title": "تخفیف ویژه آخر هفته",
    "description": "تخفیف 20 درصدی برای سفارشات آخر هفته",
    "start_time": "18:00",
    "end_time": "23:59",
    "date": "2024-01-15",
    "percent": 20,
    "status": "inactive",
    "created_at": "2024-01-10T10:00:00.000000Z",
    "updated_at": "2024-01-10T12:00:00.000000Z",
    "branch": {
      "id": 5,
      "title": "رستوران مرکزی"
    },
    "area": {
      "id": 10,
      "title": "منطقه 1"
    }
  },
  "status": true
}
```

## Error Responses

### Validation Error (400)
```json
{
  "message": "خطا در اعتبارسنجی",
  "data": {
    "branch_id": ["شناسه شعبه الزامی است"],
    "start_time": ["زمان شروع الزامی است"]
  },
  "status": false
}
```

### Not Found Error (404)
```json
{
  "message": "شرایط ویژه یافت نشد",
  "data": [],
  "status": false
}
```

### Server Error (500)
```json
{
  "message": "خطا در ایجاد شرایط ویژه",
  "data": [],
  "status": false
}
```

### Time Conflict Error
```json
{
  "message": "خطا در اعتبارسنجی",
  "data": {
    "time_conflict": ["شرایط ویژه با زمان و منطقه مشابه قبلاً ثبت شده است"]
  },
  "status": false
}
```

## Business Rules

1. **Time Conflict Prevention**: The system prevents creating special conditions that overlap in time for the same branch and area on the same date.

2. **Date Validation**: Special conditions can only be created for today or future dates.

3. **Time Validation**: End time must be after start time.

4. **Percent Range**: Discount percentage must be between 0 and 100.

5. **Status Management**: Conditions can be toggled between active and inactive states.

6. **Soft Delete**: Deleted conditions are soft deleted and can be restored if needed.

## Usage Examples

### Creating a Weekend Special Offer
```bash
curl -X POST "https://api.limoofood.com/api/v1/admin/area-special-conditions/store" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "branch_id": 5,
    "area_id": 10,
    "title": "تخفیف آخر هفته",
    "description": "تخفیف 15 درصدی برای سفارشات آخر هفته",
    "start_time": "18:00",
    "end_time": "23:59",
    "date": "2024-01-20",
    "percent": 15,
    "status": "active"
  }'
```

### Getting Active Conditions for Today
```bash
curl -X GET "https://api.limoofood.com/api/v1/admin/area-special-conditions/currently-active" \
  -H "Authorization: Bearer YOUR_TOKEN"
```

### Updating a Special Condition
```bash
curl -X PUT "https://api.limoofood.com/api/v1/admin/area-special-conditions/update" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "id": 1,
    "branch_id": 5,
    "area_id": 10,
    "title": "تخفیف ویژه آخر هفته - به‌روزرسانی شده",
    "description": "تخفیف 25 درصدی برای سفارشات آخر هفته",
    "start_time": "17:00",
    "end_time": "23:59",
    "date": "2024-01-20",
    "percent": 25,
    "status": "active"
  }'
```

## Notes

- All timestamps are in UTC format
- Time fields use 24-hour format (HH:MM)
- Date fields use YYYY-MM-DD format
- The system automatically handles timezone conversions based on the branch's location
- Special conditions are automatically deactivated after their end time
- The API supports both Persian and English error messages
