# 📱 مستند طراحی فرانت‌اند - مدیریت شماره‌های نوتیفیکیشن

## 🎯 هدف کلی
این مستند برای طراحی رابط کاربری مدیریت شماره‌های نوتیفیکیشن و پیامک رستوران‌ها و ادمین‌ها تهیه شده است.

## 📊 ساختار داده‌ها

### جدول اصلی: notification_mobiles
```typescript
interface NotificationMobile {
  id: number;
  branch_id: number;
  type: NotificationType;
  mobile: string;
  is_active: boolean;
  created_at: string;
  updated_at: string;
  branch?: {
    id: number;
    name: string;
  };
}
```

### انواع نوتیفیکیشن
```typescript
enum NotificationType {
  ORDER_NEW = 'order_new',
  ORDER_CANCELLED = 'order_cancelled',
  ORDER_READY = 'order_ready',
  TICKET_NEW = 'ticket_new',
  TICKET_ANSWERED = 'ticket_answered',
  DRIVER_ASSIGNED = 'driver_assigned',
  PAYMENT_RECEIVED = 'payment_received',
  SYSTEM_ALERT = 'system_alert'
}

interface NotificationTypeInfo {
  value: string;
  label: string;
  description: string;
}
```

## 🎨 صفحات مورد نیاز

### 1. صفحه لیست شماره‌های نوتیفیکیشن

**مسیر:** `/admin/notification-mobiles`

**عناصر صفحه:**
- جدول با ستون‌های: ID، شعبه، نوع، شماره موبایل، وضعیت، تاریخ ایجاد، عملیات
- دکمه "افزودن شماره جدید"
- فیلترهای: شعبه، نوع، وضعیت فعال/غیرفعال، جستجو در شماره
- صفحه‌بندی

**جدول:**
```typescript
const columns = [
  { key: 'id', title: 'شناسه', width: '80px' },
  { key: 'branch.name', title: 'شعبه', width: '200px' },
  { key: 'type', title: 'نوع', width: '150px', render: (type) => getTypeLabel(type) },
  { key: 'mobile', title: 'شماره موبایل', width: '150px' },
  { key: 'is_active', title: 'وضعیت', width: '100px', render: (active) => active ? 'فعال' : 'غیرفعال' },
  { key: 'created_at', title: 'تاریخ ایجاد', width: '150px', render: (date) => formatDate(date) },
  { key: 'actions', title: 'عملیات', width: '150px' }
];
```

**فیلترها:**
```typescript
const filters = {
  branch_id: { type: 'select', placeholder: 'انتخاب شعبه', options: branches },
  type: { type: 'select', placeholder: 'انتخاب نوع', options: notificationTypes },
  is_active: { type: 'select', placeholder: 'وضعیت', options: [{value: true, label: 'فعال'}, {value: false, label: 'غیرفعال'}] },
  mobile: { type: 'text', placeholder: 'جستجو در شماره موبایل' }
};
```

### 2. صفحه افزودن/ویرایش شماره

**مسیر:** `/admin/notification-mobiles/create` یا `/admin/notification-mobiles/edit/:id`

**فرم:**
```typescript
const formFields = [
  {
    name: 'branch_id',
    type: 'select',
    label: 'شعبه',
    required: true,
    options: branches,
    validation: { required: 'انتخاب شعبه الزامی است' }
  },
  {
    name: 'type',
    type: 'select',
    label: 'نوع نوتیفیکیشن',
    required: true,
    options: notificationTypes,
    validation: { required: 'انتخاب نوع الزامی است' }
  },
  {
    name: 'mobile',
    type: 'text',
    label: 'شماره موبایل',
    required: true,
    placeholder: '09123456789',
    validation: { 
      required: 'شماره موبایل الزامی است',
      pattern: /^09[0-9]{9}$/,
      message: 'فرمت شماره موبایل نامعتبر است'
    }
  },
  {
    name: 'is_active',
    type: 'switch',
    label: 'وضعیت فعال',
    default: true
  }
];
```

### 3. صفحه جزئیات شماره

**مسیر:** `/admin/notification-mobiles/view/:id`

**اطلاعات نمایشی:**
- شناسه
- نام شعبه
- نوع نوتیفیکیشن (با برچسب فارسی)
- شماره موبایل
- وضعیت فعال/غیرفعال
- تاریخ ایجاد
- تاریخ آخرین بروزرسانی

## 🔧 API Integration

### Base URL
```
/api/v1/admin/notification-mobiles
```

### Endpoints

#### 1. دریافت لیست
```typescript
GET /api/v1/admin/notification-mobiles
Query: {
  branch_id?: number,
  type?: string,
  is_active?: boolean,
  mobile?: string,
  page?: number,
  per_page?: number
}

Response: {
  data: NotificationMobile[],
  pagination: {
    current_page: number,
    last_page: number,
    per_page: number,
    total: number
  }
}
```

#### 2. ایجاد جدید
```typescript
POST /api/v1/admin/notification-mobiles/store
Body: {
  branch_id: number,
  type: string,
  mobile: string,
  is_active?: boolean
}

Response: {
  data: NotificationMobile,
  message: string
}
```

#### 3. بروزرسانی
```typescript
PUT /api/v1/admin/notification-mobiles/update
Body: {
  notification_mobile_id: number,
  branch_id: number,
  type: string,
  mobile: string,
  is_active?: boolean
}

Response: {
  data: NotificationMobile,
  message: string
}
```

#### 4. حذف
```typescript
DELETE /api/v1/admin/notification-mobiles/destroy
Body: {
  notification_mobile_id: number
}

Response: {
  message: string
}
```

#### 5. دریافت بر اساس شعبه
```typescript
GET /api/v1/admin/notification-mobiles/by-branch?branch_id=5
Response: {
  data: NotificationMobile[]
}
```

#### 6. دریافت بر اساس نوع
```typescript
GET /api/v1/admin/notification-mobiles/by-type?type=order_new
Response: {
  data: NotificationMobile[]
}
```

#### 7. دریافت شماره‌های موبایل
```typescript
GET /api/v1/admin/notification-mobiles/mobile-numbers?branch_id=5&type=order_new&active_only=true
Response: {
  data: {
    mobile_numbers: string[],
    count: number
  }
}
```

#### 8. دریافت انواع نوتیفیکیشن
```typescript
GET /api/v1/admin/notification-mobiles/notification-types
Response: {
  data: NotificationTypeInfo[]
}
```

## 🎨 UI/UX Guidelines

### رنگ‌بندی
```css
:root {
  --primary-color: #2563eb;
  --success-color: #10b981;
  --warning-color: #f59e0b;
  --danger-color: #ef4444;
  --gray-50: #f9fafb;
  --gray-100: #f3f4f6;
  --gray-500: #6b7280;
  --gray-900: #111827;
}
```

### کامپوننت‌های مورد نیاز

#### 1. جدول شماره‌های نوتیفیکیشن
```typescript
interface NotificationMobileTableProps {
  data: NotificationMobile[];
  loading: boolean;
  onEdit: (id: number) => void;
  onDelete: (id: number) => void;
  onToggleStatus: (id: number, status: boolean) => void;
}
```

#### 2. فرم افزودن/ویرایش
```typescript
interface NotificationMobileFormProps {
  initialData?: NotificationMobile;
  branches: Branch[];
  notificationTypes: NotificationTypeInfo[];
  onSubmit: (data: NotificationMobileFormData) => void;
  loading: boolean;
}
```

#### 3. فیلترها
```typescript
interface NotificationMobileFiltersProps {
  branches: Branch[];
  notificationTypes: NotificationTypeInfo[];
  onFilterChange: (filters: FilterData) => void;
  onReset: () => void;
}
```

#### 4. مودال حذف
```typescript
interface DeleteModalProps {
  isOpen: boolean;
  onClose: () => void;
  onConfirm: () => void;
  title: string;
  message: string;
  loading: boolean;
}
```

### آیکون‌های مورد نیاز
- 📱 آیکون موبایل
- 🔔 آیکون نوتیفیکیشن
- ➕ آیکون افزودن
- ✏️ آیکون ویرایش
- 🗑️ آیکون حذف
- ✅ آیکون فعال
- ❌ آیکون غیرفعال
- 🔍 آیکون جستجو
- 📊 آیکون فیلتر

## 📱 Responsive Design

### Breakpoints
```css
/* Mobile */
@media (max-width: 768px) {
  .table-container {
    overflow-x: auto;
  }
  
  .form-grid {
    grid-template-columns: 1fr;
  }
}

/* Tablet */
@media (min-width: 769px) and (max-width: 1024px) {
  .form-grid {
    grid-template-columns: 1fr 1fr;
  }
}

/* Desktop */
@media (min-width: 1025px) {
  .form-grid {
    grid-template-columns: 1fr 1fr 1fr;
  }
}
```

## 🔄 State Management

### Redux/Zustand Store Structure
```typescript
interface NotificationMobileState {
  // Data
  notificationMobiles: NotificationMobile[];
  branches: Branch[];
  notificationTypes: NotificationTypeInfo[];
  
  // UI State
  loading: boolean;
  error: string | null;
  filters: FilterData;
  pagination: PaginationData;
  
  // Actions
  fetchNotificationMobiles: () => Promise<void>;
  createNotificationMobile: (data: NotificationMobileFormData) => Promise<void>;
  updateNotificationMobile: (id: number, data: NotificationMobileFormData) => Promise<void>;
  deleteNotificationMobile: (id: number) => Promise<void>;
  setFilters: (filters: FilterData) => void;
  resetFilters: () => void;
}
```

## 🧪 Validation Rules

### Client-side Validation
```typescript
const validationRules = {
  branch_id: {
    required: 'انتخاب شعبه الزامی است',
    min: { value: 1, message: 'شناسه شعبه نامعتبر است' }
  },
  type: {
    required: 'انتخاب نوع الزامی است',
    enum: { 
      values: Object.values(NotificationType), 
      message: 'نوع نوتیفیکیشن نامعتبر است' 
    }
  },
  mobile: {
    required: 'شماره موبایل الزامی است',
    pattern: {
      value: /^09[0-9]{9}$/,
      message: 'فرمت شماره موبایل نامعتبر است (مثال: 09123456789)'
    },
    minLength: { value: 11, message: 'شماره موبایل باید 11 رقم باشد' },
    maxLength: { value: 11, message: 'شماره موبایل باید 11 رقم باشد' }
  },
  is_active: {
    type: 'boolean'
  }
};
```

## 🎯 User Stories

### 1. مدیر سیستم
- می‌خواهد لیست تمام شماره‌های نوتیفیکیشن را ببیند
- می‌خواهد شماره جدید برای هر شعبه و نوع اضافه کند
- می‌خواهد شماره‌های موجود را ویرایش کند
- می‌خواهد شماره‌ها را فعال/غیرفعال کند
- می‌خواهد شماره‌های غیرضروری را حذف کند

### 2. مدیر شعبه
- می‌خواهد شماره‌های مربوط به شعبه خود را ببیند
- می‌خواهد شماره‌های مختلف برای انواع مختلف پیام تنظیم کند
- می‌خواهد وضعیت شماره‌ها را مدیریت کند

## 🚀 Implementation Checklist

### Phase 1: Core Features
- [ ] صفحه لیست شماره‌های نوتیفیکیشن
- [ ] فرم افزودن شماره جدید
- [ ] فرم ویرایش شماره موجود
- [ ] مودال حذف شماره
- [ ] فیلترهای پایه (شعبه، نوع، وضعیت)

### Phase 2: Advanced Features
- [ ] جستجو در شماره‌های موبایل
- [ ] مرتب‌سازی جدول
- [ ] Export به Excel
- [ ] Bulk operations (حذف چندتایی، تغییر وضعیت چندتایی)
- [ ] تاریخچه تغییرات

### Phase 3: UX Enhancements
- [ ] Loading states
- [ ] Error handling
- [ ] Success notifications
- [ ] Confirmation dialogs
- [ ] Keyboard shortcuts
- [ ] Dark mode support

## 📋 Sample Data

### نمونه داده‌های تست
```json
[
  {
    "id": 1,
    "branch_id": 5,
    "type": "order_new",
    "mobile": "09123456789",
    "is_active": true,
    "created_at": "2025-10-11T22:20:32.000000Z",
    "updated_at": "2025-10-11T22:20:32.000000Z",
    "branch": {
      "id": 5,
      "name": "رستوران نمونه"
    }
  },
  {
    "id": 2,
    "branch_id": 5,
    "type": "ticket_new",
    "mobile": "09987654321",
    "is_active": false,
    "created_at": "2025-10-11T22:25:15.000000Z",
    "updated_at": "2025-10-11T22:30:45.000000Z",
    "branch": {
      "id": 5,
      "name": "رستوران نمونه"
    }
  }
]
```

### نمونه انواع نوتیفیکیشن
```json
[
  {
    "value": "order_new",
    "label": "سفارش جدید",
    "description": "پیامک سفارش جدید برای رستوران"
  },
  {
    "value": "order_cancelled",
    "label": "لغو سفارش",
    "description": "پیامک لغو سفارش برای رستوران"
  },
  {
    "value": "order_ready",
    "label": "آماده شدن سفارش",
    "description": "پیامک آماده شدن سفارش برای راننده"
  },
  {
    "value": "ticket_new",
    "label": "تیکت جدید",
    "description": "پیامک تیکت جدید برای ادمین"
  },
  {
    "value": "ticket_answered",
    "label": "پاسخ تیکت",
    "description": "پیامک پاسخ تیکت برای ادمین"
  },
  {
    "value": "driver_assigned",
    "label": "انتساب راننده",
    "description": "پیامک انتساب راننده برای رستوران"
  },
  {
    "value": "payment_received",
    "label": "دریافت پرداخت",
    "description": "پیامک دریافت پرداخت برای رستوران"
  },
  {
    "value": "system_alert",
    "label": "هشدار سیستم",
    "description": "پیامک هشدارهای سیستم برای ادمین"
  }
]
```

## 🔐 Security Considerations

### Authentication
- تمام API calls نیاز به Bearer token دارند
- Token باید در header `Authorization` ارسال شود

### Authorization
- فقط کاربران با نقش admin دسترسی دارند
- امکان محدود کردن دسترسی بر اساس شعبه (در آینده)

### Input Validation
- تمام input ها در سمت کلاینت و سرور validate می‌شوند
- شماره موبایل باید با الگوی ایرانی مطابقت داشته باشد
- جلوگیری از SQL Injection و XSS

## 📞 Support & Contact

برای سوالات فنی یا مشکلات پیاده‌سازی با تیم بک‌اند تماس بگیرید.

---

**تاریخ ایجاد:** 2025-10-11  
**نسخه:** 1.0  
**آخرین بروزرسانی:** 2025-10-11
