# API مدیریت شماره‌های نوتیفیکیشن و پیامک

## 📋 توضیحات کلی

این API برای مدیریت شماره‌های موبایل مربوط به نوتیفیکیشن‌ها و پیامک‌های رستوران و ادمین طراحی شده است.

## 🗂️ ساختار جدول

```sql
notification_mobiles:
- id (BIGINT PRIMARY KEY)
- branch_id (BIGINT) - شناسه شعبه
- type (ENUM) - نوع نوتیفیکیشن
- mobile (VARCHAR(20)) - شماره موبایل
- is_active (BOOLEAN) - وضعیت فعال/غیرفعال
- created_at, updated_at (TIMESTAMP)
```

## 📱 انواع نوتیفیکیشن

| نوع | برچسب فارسی | توضیحات |
|-----|-------------|---------|
| `order_new` | سفارش جدید | پیامک سفارش جدید برای رستوران |
| `order_cancelled` | لغو سفارش | پیامک لغو سفارش برای رستوران |
| `order_ready` | آماده شدن سفارش | پیامک آماده شدن سفارش برای راننده |
| `ticket_new` | تیکت جدید | پیامک تیکت جدید برای ادمین |
| `ticket_answered` | پاسخ تیکت | پیامک پاسخ تیکت برای ادمین |
| `driver_assigned` | انتساب راننده | پیامک انتساب راننده برای رستوران |
| `payment_received` | دریافت پرداخت | پیامک دریافت پرداخت برای رستوران |
| `system_alert` | هشدار سیستم | پیامک هشدارهای سیستم برای ادمین |

## 🚀 API Endpoints

### 1. دریافت لیست شماره‌های نوتیفیکیشن
```http
GET /api/v1/admin/notification-mobiles
```

**Query Parameters:**
- `branch_id` (optional): فیلتر بر اساس شعبه
- `type` (optional): فیلتر بر اساس نوع
- `is_active` (optional): فیلتر بر اساس وضعیت فعال
- `mobile` (optional): جستجو در شماره‌ها
- `page` (optional): شماره صفحه
- `per_page` (optional): تعداد آیتم در هر صفحه

**Response:**
```json
{
  "data": [
    {
      "id": 1,
      "branch_id": 5,
      "type": "order_new",
      "mobile": "09123456789",
      "is_active": true,
      "created_at": "2025-10-11T22:20:32.000000Z",
      "updated_at": "2025-10-11T22:20:32.000000Z",
      "branch": {
        "id": 5,
        "name": "رستوران نمونه"
      }
    }
  ],
  "pagination": {
    "current_page": 1,
    "last_page": 1,
    "per_page": 15,
    "total": 1
  }
}
```

### 2. ایجاد شماره نوتیفیکیشن جدید
```http
POST /api/v1/admin/notification-mobiles/store
```

**Body:**
```json
{
  "branch_id": 5,
  "type": "order_new",
  "mobile": "09123456789",
  "is_active": true
}
```

**Response:**
```json
{
  "data": {
    "id": 1,
    "branch_id": 5,
    "type": "order_new",
    "mobile": "09123456789",
    "is_active": true,
    "created_at": "2025-10-11T22:20:32.000000Z",
    "updated_at": "2025-10-11T22:20:32.000000Z",
    "branch": {
      "id": 5,
      "name": "رستوران نمونه"
    }
  },
  "message": "شماره نوتیفیکیشن با موفقیت ایجاد شد"
}
```

### 3. بروزرسانی شماره نوتیفیکیشن
```http
PUT /api/v1/admin/notification-mobiles/update
```

**Body:**
```json
{
  "notification_mobile_id": 1,
  "branch_id": 5,
  "type": "order_new",
  "mobile": "09987654321",
  "is_active": false
}
```

### 4. حذف شماره نوتیفیکیشن
```http
DELETE /api/v1/admin/notification-mobiles/destroy
```

**Body:**
```json
{
  "notification_mobile_id": 1
}
```

### 5. دریافت شماره‌های یک شعبه
```http
GET /api/v1/admin/notification-mobiles/by-branch?branch_id=5
```

### 6. دریافت شماره‌های یک نوع
```http
GET /api/v1/admin/notification-mobiles/by-type?type=order_new
```

### 7. دریافت شماره‌های موبایل برای ارسال پیامک
```http
GET /api/v1/admin/notification-mobiles/mobile-numbers?branch_id=5&type=order_new&active_only=true
```

**Response:**
```json
{
  "data": {
    "mobile_numbers": ["09123456789", "09987654321"],
    "count": 2
  }
}
```

### 8. دریافت انواع نوتیفیکیشن
```http
GET /api/v1/admin/notification-mobiles/notification-types
```

**Response:**
```json
{
  "data": [
    {
      "value": "order_new",
      "label": "سفارش جدید",
      "description": "پیامک سفارش جدید برای رستوران"
    },
    {
      "value": "order_cancelled",
      "label": "لغو سفارش",
      "description": "پیامک لغو سفارش برای رستوران"
    }
  ]
}
```

## 💡 مثال‌های کاربردی

### مثال 1: تنظیم شماره برای سفارش‌های جدید یک رستوران
```bash
curl -X POST "http://localhost/api/v1/admin/notification-mobiles/store" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "branch_id": 5,
    "type": "order_new",
    "mobile": "09123456789",
    "is_active": true
  }'
```

### مثال 2: دریافت شماره‌های فعال برای ارسال پیامک سفارش جدید
```bash
curl -X GET "http://localhost/api/v1/admin/notification-mobiles/mobile-numbers?branch_id=5&type=order_new&active_only=true" \
  -H "Authorization: Bearer YOUR_TOKEN"
```

### مثال 3: دریافت تمام شماره‌های یک شعبه
```bash
curl -X GET "http://localhost/api/v1/admin/notification-mobiles/by-branch?branch_id=5" \
  -H "Authorization: Bearer YOUR_TOKEN"
```

## 🔧 استفاده در کد

### دریافت شماره‌های موبایل در کد:
```php
use App\Models\NotificationMobile;
use App\Enums\NotificationType;

// دریافت شماره‌های فعال برای سفارش جدید یک شعبه
$mobileNumbers = NotificationMobile::getMobilesForBranchAndType(
    $branchId, 
    NotificationType::ORDER_NEW
);

// دریافت شماره‌های فعال برای تیکت جدید (همه شعب)
$adminMobiles = NotificationMobile::getMobilesForType(
    NotificationType::TICKET_NEW
);
```

### ارسال پیامک با استفاده از شماره‌های تنظیم شده:
```php
use App\Repositories\NotificationMobile\NotificationMobileRepositoryInterface;
use App\Enums\NotificationType;

$notificationRepo = app(NotificationMobileRepositoryInterface::class);

// دریافت شماره‌های موبایل برای ارسال پیامک
$mobileNumbers = $notificationRepo->getActiveMobileNumbers(
    $branchId, 
    NotificationType::ORDER_NEW
);

// ارسال پیامک به تمام شماره‌ها
foreach ($mobileNumbers as $mobile) {
    $smsRepository->sendSMSNewOrder('#' . $orderId, $mobile);
}
```

## ⚠️ نکات مهم

1. **فرمت شماره موبایل**: باید با الگوی `09[0-9]{9}` مطابقت داشته باشد
2. **یکتایی**: ترکیب `branch_id + type + mobile` باید یکتا باشد
3. **Foreign Key**: `branch_id` باید در جدول `branches` موجود باشد
4. **Enum Values**: `type` باید یکی از مقادیر تعریف شده در enum باشد
5. **Authentication**: تمام endpoint ها نیاز به احراز هویت دارند

## 🎯 مزایای این سیستم

- **انعطاف‌پذیری**: هر شعبه می‌تواند شماره‌های مختلف برای انواع مختلف تنظیم کند
- **مدیریت متمرکز**: تمام شماره‌ها در یک مکان مدیریت می‌شوند
- **قابلیت توسعه**: به راحتی می‌توان انواع جدید پیام اضافه کرد
- **عملکرد بهتر**: استفاده از index برای جستجوی سریع
- **سازگاری**: با ساختار موجود سیستم سازگار است
