# SMS Settings API Documentation

## Overview
This API provides CRUD operations for managing SMS settings in the system. All endpoints are under the admin section and require proper authentication.

## Base URL
```
/v1/admin/sms-settings
```

## Authentication
All endpoints require authentication. Include the authorization token in the request headers:
```
Authorization: Bearer {your_token}
```

## Endpoints

### 1. Get All SMS Settings
**GET** `/v1/admin/sms-settings/`

Retrieves a list of all SMS settings with optional filtering.

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `search` | string | No | Search in title and repository fields |
| `status` | boolean | No | Filter by status (true/false) |
| `repository` | string | No | Filter by specific repository |
| `paginate` | boolean | No | Enable pagination (default: false) |
| `per_page` | integer | No | Number of items per page (default: 15) |

#### Example Request
```http
GET /v1/admin/sms-settings/?search=kavenegar&status=true&paginate=true&per_page=10
Authorization: Bearer {your_token}
```

#### Example Response
```json
{
  "data": [
    {
      "id": 1,
      "title": "Kavenegar SMS Service",
      "repository": "kavenegar",
      "status": true,
      "created_at": "2025-01-12T10:30:00.000000Z",
      "updated_at": "2025-01-12T10:30:00.000000Z",
      "branch": null
    }
  ]
}
```

---

### 2. Get Single SMS Setting
**GET** `/v1/admin/sms-settings/show`

Retrieves a specific SMS setting by ID.

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `sms_setting_id` | integer | Yes | ID of the SMS setting |

#### Example Request
```http
GET /v1/admin/sms-settings/show?sms_setting_id=1
Authorization: Bearer {your_token}
```

#### Example Response
```json
{
  "data": {
    "id": 1,
    "title": "Kavenegar SMS Service",
    "repository": "kavenegar",
    "status": true,
    "created_at": "2025-01-12T10:30:00.000000Z",
    "updated_at": "2025-01-12T10:30:00.000000Z",
    "branch": null
  }
}
```

#### Error Response
```json
{
  "error": "شناسه تنظیمات پیامک الزامی است"
}
```

---

### 3. Create SMS Setting
**POST** `/v1/admin/sms-settings/store`

Creates a new SMS setting.

#### Request Body
| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `title` | string | Yes | Title of the SMS setting (max: 255 chars) |
| `repository` | string | Yes | Repository/service name (max: 255 chars) |
| `status` | boolean | No | Status of the setting (default: true) |

#### Example Request
```http
POST /v1/admin/sms-settings/store
Authorization: Bearer {your_token}
Content-Type: application/json

{
  "title": "Melipayamak SMS Service",
  "repository": "melipayamak",
  "status": true
}
```

#### Example Response
```json
{
  "data": {
    "id": 2,
    "title": "Melipayamak SMS Service",
    "repository": "melipayamak",
    "status": true,
    "created_at": "2025-01-12T11:00:00.000000Z",
    "updated_at": "2025-01-12T11:00:00.000000Z"
  },
  "message": "تنظیمات پیامک با موفقیت ایجاد شد"
}
```

#### Error Response
```json
{
  "error": "عنوان الزامی است"
}
```

---

### 4. Update SMS Setting
**PUT** `/v1/admin/sms-settings/update`

Updates an existing SMS setting.

#### Request Body
| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `sms_setting_id` | integer | Yes | ID of the SMS setting to update |
| `title` | string | Yes | Title of the SMS setting (max: 255 chars) |
| `repository` | string | Yes | Repository/service name (max: 255 chars) |
| `status` | boolean | No | Status of the setting |

#### Example Request
```http
PUT /v1/admin/sms-settings/update
Authorization: Bearer {your_token}
Content-Type: application/json

{
  "sms_setting_id": 1,
  "title": "Updated Kavenegar SMS Service",
  "repository": "kavenegar",
  "status": false
}
```

#### Example Response
```json
{
  "data": {
    "id": 1,
    "title": "Updated Kavenegar SMS Service",
    "repository": "kavenegar",
    "status": false,
    "created_at": "2025-01-12T10:30:00.000000Z",
    "updated_at": "2025-01-12T11:15:00.000000Z"
  },
  "message": "تنظیمات پیامک با موفقیت بروزرسانی شد"
}
```

#### Error Response
```json
{
  "error": "شناسه تنظیمات پیامک الزامی است"
}
```

---

### 5. Delete SMS Setting
**DELETE** `/v1/admin/sms-settings/destroy`

Deletes an SMS setting.

#### Request Body
| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `sms_setting_id` | integer | Yes | ID of the SMS setting to delete |

#### Example Request
```http
DELETE /v1/admin/sms-settings/destroy
Authorization: Bearer {your_token}
Content-Type: application/json

{
  "sms_setting_id": 1
}
```

#### Example Response
```json
{
  "message": "تنظیمات پیامک با موفقیت حذف شد"
}
```

#### Error Response
```json
{
  "error": "شناسه تنظیمات پیامک الزامی است"
}
```

---

### 6. Get SMS Settings by Repository
**GET** `/v1/admin/sms-settings/by-repository`

Retrieves SMS settings filtered by repository name.

#### Query Parameters
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `repository` | string | Yes | Repository name to filter by |

#### Example Request
```http
GET /v1/admin/sms-settings/by-repository?repository=kavenegar
Authorization: Bearer {your_token}
```

#### Example Response
```json
{
  "data": [
    {
      "id": 1,
      "title": "Kavenegar SMS Service",
      "repository": "kavenegar",
      "status": true,
      "created_at": "2025-01-12T10:30:00.000000Z",
      "updated_at": "2025-01-12T10:30:00.000000Z"
    }
  ]
}
```

---

### 7. Get Active SMS Settings
**GET** `/v1/admin/sms-settings/active`

Retrieves all active SMS settings (status = true).

#### Example Request
```http
GET /v1/admin/sms-settings/active
Authorization: Bearer {your_token}
```

#### Example Response
```json
{
  "data": [
    {
      "id": 1,
      "title": "Kavenegar SMS Service",
      "repository": "kavenegar",
      "status": true,
      "created_at": "2025-01-12T10:30:00.000000Z",
      "updated_at": "2025-01-12T10:30:00.000000Z"
    },
    {
      "id": 2,
      "title": "Melipayamak SMS Service",
      "repository": "melipayamak",
      "status": true,
      "created_at": "2025-01-12T11:00:00.000000Z",
      "updated_at": "2025-01-12T11:00:00.000000Z"
    }
  ]
}
```

---

## Error Codes

| HTTP Status | Description |
|-------------|-------------|
| 200 | Success |
| 201 | Created successfully |
| 400 | Bad Request - Validation error |
| 404 | Not Found - Resource not found |
| 500 | Internal Server Error |

## Common Error Messages

| Message | Description |
|---------|-------------|
| `عنوان الزامی است` | Title is required |
| `مخزن الزامی است` | Repository is required |
| `شناسه تنظیمات پیامک الزامی است` | SMS setting ID is required |
| `تنظیمات پیامک یافت نشد` | SMS setting not found |
| `خطا در ایجاد تنظیمات پیامک` | Error creating SMS setting |
| `خطا در بروزرسانی تنظیمات پیامک` | Error updating SMS setting |
| `خطا در حذف تنظیمات پیامک` | Error deleting SMS setting |

## Data Model

### SmsSetting Object
```json
{
  "id": "integer",
  "title": "string",
  "repository": "string", 
  "status": "boolean",
  "created_at": "datetime",
  "updated_at": "datetime",
  "branch": "object|null"
}
```

## Frontend Integration Examples

### JavaScript/Axios Example
```javascript
// Get all SMS settings
const getSmsSettings = async () => {
  try {
    const response = await axios.get('/v1/admin/sms-settings/', {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    });
    return response.data.data;
  } catch (error) {
    console.error('Error fetching SMS settings:', error.response.data);
  }
};

// Create new SMS setting
const createSmsSetting = async (data) => {
  try {
    const response = await axios.post('/v1/admin/sms-settings/store', data, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    return response.data;
  } catch (error) {
    console.error('Error creating SMS setting:', error.response.data);
  }
};

// Update SMS setting
const updateSmsSetting = async (id, data) => {
  try {
    const response = await axios.put('/v1/admin/sms-settings/update', {
      sms_setting_id: id,
      ...data
    }, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    return response.data;
  } catch (error) {
    console.error('Error updating SMS setting:', error.response.data);
  }
};

// Delete SMS setting
const deleteSmsSetting = async (id) => {
  try {
    const response = await axios.delete('/v1/admin/sms-settings/destroy', {
      data: { sms_setting_id: id },
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    return response.data;
  } catch (error) {
    console.error('Error deleting SMS setting:', error.response.data);
  }
};
```

### React Hook Example
```javascript
import { useState, useEffect } from 'react';
import axios from 'axios';

const useSmsSettings = () => {
  const [smsSettings, setSmsSettings] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);

  const fetchSmsSettings = async (filters = {}) => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.get('/v1/admin/sms-settings/', {
        params: filters,
        headers: { 'Authorization': `Bearer ${token}` }
      });
      setSmsSettings(response.data.data);
    } catch (err) {
      setError(err.response?.data?.error || 'Error fetching SMS settings');
    } finally {
      setLoading(false);
    }
  };

  const createSmsSetting = async (data) => {
    try {
      const response = await axios.post('/v1/admin/sms-settings/store', data, {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      setSmsSettings(prev => [...prev, response.data.data]);
      return response.data;
    } catch (err) {
      setError(err.response?.data?.error || 'Error creating SMS setting');
      throw err;
    }
  };

  const updateSmsSetting = async (id, data) => {
    try {
      const response = await axios.put('/v1/admin/sms-settings/update', {
        sms_setting_id: id,
        ...data
      }, {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      setSmsSettings(prev => 
        prev.map(setting => 
          setting.id === id ? response.data.data : setting
        )
      );
      return response.data;
    } catch (err) {
      setError(err.response?.data?.error || 'Error updating SMS setting');
      throw err;
    }
  };

  const deleteSmsSetting = async (id) => {
    try {
      await axios.delete('/v1/admin/sms-settings/destroy', {
        data: { sms_setting_id: id },
        headers: { 'Authorization': `Bearer ${token}` }
      });
      setSmsSettings(prev => prev.filter(setting => setting.id !== id));
    } catch (err) {
      setError(err.response?.data?.error || 'Error deleting SMS setting');
      throw err;
    }
  };

  return {
    smsSettings,
    loading,
    error,
    fetchSmsSettings,
    createSmsSetting,
    updateSmsSetting,
    deleteSmsSetting
  };
};

export default useSmsSettings;
```

## Notes
- All timestamps are in UTC format
- The API returns Persian error messages
- All endpoints require proper authentication
- The `branch` relationship is included in responses but may be null
- Status field accepts boolean values (true/false)
- Repository field should contain the service provider name (e.g., 'kavenegar', 'melipayamak')
